package com.kontakt.sdk.android.cloud.api.executor.managers;

import com.kontakt.sdk.android.cloud.api.ManagersApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ManagersService;
import com.kontakt.sdk.android.cloud.response.paginated.Managers;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link ManagersApi}. Use this class if you want to fetch
 * managers through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Managers managers = kontaktCloud.managers().fetch()
 *      .subordinatesOf(supervisorID)
 *      .execute();
 *   </code>
 * </pre>
 */
public class ManagersRequestExecutor extends PaginatedRequestExecutor<Managers> {

  private final ManagersService managersService;
  private UUID managerId;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param managersService the managers API service.
   */
  public ManagersRequestExecutor(final ManagersService managersService) {
    this.managersService = managersService;
  }

  /**
   * We can use this method if we want to fetch all subordinate managers for the given manager.
   *
   * @param managerId the manager's unique identifier.
   * @return this request executor.
   */
  public ManagersRequestExecutor subordinatesOf(final UUID managerId) {
    SDKPreconditions.checkNotNull(managerId, "manager ID cannot be null");
    this.managerId = managerId;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ManagersRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ManagersRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ManagersRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ManagersRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ManagersRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Managers> prepareCall() {
    final Call<Managers> call;
    if (eTag != null) {
      if (managerId != null) {
        call = managersService.getManagerSubOrdinates(managerId.toString(), params(), eTag);
      } else {
        call = managersService.getManagers(params(), eTag);
      }
    } else {
      if (managerId != null) {
        call = managersService.getManagerSubOrdinates(managerId.toString(), params());
      } else {
        call = managersService.getManagers(params());
      }
    }
    return call;
  }
}
