package com.kontakt.sdk.android.cloud.api.executor.devices;

import android.util.Base64;
import android.util.Log;
import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.ConversionUtils.hexStringToByteArray;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to decrypt encrypted Eddystone TLM frame.
 * Both EID and ETLM frames are required for this operation.
 * Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.devices().decryptTelemetry("eidFrame", "eTlmFrame).execute()
 *   </code>
 * </pre>
 */
public class TelemetryRequestExecutor extends RequestExecutor<Telemetry> {

  private final DevicesService service;
  private final String eid;
  private final String etlmFrame;

  public TelemetryRequestExecutor(final DevicesService service, final String eid, final String etlmFrame) {
    this.service = service;
    this.eid = eid;
    this.etlmFrame = etlmFrame;
  }

  @Override
  protected Call<Telemetry> prepareCall() {
    return service.decryptTelemetry(params());
  }

  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    if (etlmFrame != null) {
      String encoded = Base64.encodeToString(hexStringToByteArray(etlmFrame.trim()), Base64.NO_WRAP);
      Log.d("Encoded", "params: " + encoded);
      params.put(CloudConstants.Devices.EDDYSTONE_ETLM_FRAME, encoded);
    }
    if (eid != null) {
      params.put(CloudConstants.Devices.EDDYSTONE_EID_PARAMETER, eid.trim());
    }
    return params;
  }
}
