package com.kontakt.sdk.android.ble.monitoring;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Event;
import com.kontakt.sdk.android.common.model.EventPacket;
import com.kontakt.sdk.android.common.model.EventType;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class EventCollector implements IEventCollector {

  static final String TAG = "EventCollector";

  private final EventSender eventSender;
  private ScanContext scanContext;
  private ScheduledExecutorService executorService;

  public EventCollector(EventSender eventSender, ScanContext scanContext) {
    this.eventSender = checkNotNull(eventSender);
    this.scanContext = checkNotNull(scanContext);
  }

  public void collect(RemoteBluetoothDevice device) {
    this.eventSender.collect(device);
  }

  @Override
  public void stop() {
    if (executorService == null) {
      return;
    }
    executorService.shutdown();
    executorService = null;
  }

  @Override
  public void clear() {
    this.eventSender.clear();
  }

  @Override
  public void start() {
    if (executorService != null || !scanContext.isMonitoringEnabled()) {
      return;
    }
    executorService = Executors.newSingleThreadScheduledExecutor();
    executorService.scheduleWithFixedDelay(eventSender, scanContext.getMonitoringSyncInterval(), scanContext.getMonitoringSyncInterval(),
        TimeUnit.SECONDS);
  }

  ScheduledExecutorService getExecutorService() {
    return executorService;
  }

  public static class EventSender implements Runnable {

    private static final int DEFAULT_BUFFER_SIZE = 200;
    private final ConcurrentSkipListSet<String> ignored = new ConcurrentSkipListSet<>();
    private final BlockingQueue<Event> eventsBuffer;
    private final KontaktCloud kontaktCloud;

    public EventSender(final KontaktCloud kontaktCloud) {
      this.eventsBuffer = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
      this.kontaktCloud = kontaktCloud;
    }

    @Override
    public void run() {
      final List<Event> events = new ArrayList<>();
      eventsBuffer.drainTo(events);
      if (events.isEmpty()) {
        Logger.d(TAG + " Nothing to send");
        return;
      }

      final EventPacket eventPacket = EventPacket.from(events, System.currentTimeMillis());
      try {
        kontaktCloud.events().collect(eventPacket, null).execute();
      } catch (IOException | KontaktCloudException e) {
        Throwable cause = e.getCause();
        if (UnknownHostException.class.isInstance(cause) || SocketTimeoutException.class.isInstance(cause)) {
          eventsBuffer.addAll(events);
        } else {
          Logger.e(TAG + " Error occurred when try to send monitoring events", e);
        }
      }
    }

    void collect(final RemoteBluetoothDevice device) {
      final String notificationId = device.getUniqueId() + ":" + device.getBatteryPower();
      if (ignored.contains(notificationId)) {
        return;
      }
      Event event = Event.of(device, EventType.MONITORING);
      try {
        eventsBuffer.add(event);
        ignored.add(notificationId);
      } catch (IllegalStateException e) {
        Logger.e("Event collector queue is full", e);
      }
    }

    public void clear() {
      eventsBuffer.clear();
      ignored.clear();
    }
  }
}
