package com.kontakt.sdk.android.ble.filter.eddystone;

import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Factory class producing filters for Eddystone-specific packages: the UID packet, the URL packet, the TLM packet.
 */
public final class EddystoneFilters {

  private EddystoneFilters() {
  }

  /**
   * Creates new UID filter accepting UID packet with specified namespace id and instance id.
   *
   * @param namespace the namespace id
   * @param instanceId the instance id
   * @return the uID filter
   */
  public static EddystoneFilter newUIDFilter(final String namespace, final String instanceId) {
    SDKPreconditions.checkNotNull(namespace, "Namespace Id is null");
    SDKPreconditions.checkNotNull(instanceId, "Instance Id is null");
    return new EddystoneFilter() {
      @Override
      public boolean apply(IEddystoneDevice eddystoneDevice) {
        return namespace.equals(eddystoneDevice.getNamespace()) && instanceId.equals(eddystoneDevice.getInstanceId());
      }
    };
  }

  /**
   * Creates new URL filters accepting URL packet with specified url.
   *
   * @param url the url
   * @return the uRL filter
   */
  public static EddystoneFilter newURLFilter(final String url) {
    SDKPreconditions.checkNotNull(url, "URL is null");

    return new EddystoneFilter() {
      @Override
      public boolean apply(IEddystoneDevice eddystoneDevice) {
        return url.equals(eddystoneDevice.getUrl());
      }
    };
  }

  /**
   * Creates new filter accepting UID packet with specified namespace id.
   *
   * @param namespace the namespace id
   * @return the uID filter
   */
  public static EddystoneFilter newNamespaceFilter(final String namespace) {
    SDKPreconditions.checkNotNull(namespace, "Namespace id is null");

    return new EddystoneFilter() {
      @Override
      public boolean apply(IEddystoneDevice eddystoneDevice) {
        return namespace.equals(eddystoneDevice.getNamespace());
      }
    };
  }

  /**
   * Creates new filter accepting UID packet with specified instance id.
   *
   * @param instanceId the instance id
   * @return the UID filter
   */
  public static EddystoneFilter newInstanceIdFilter(final String instanceId) {
    SDKPreconditions.checkNotNull(instanceId, "Instance id is null");

    return new EddystoneFilter() {
      @Override
      public boolean apply(IEddystoneDevice eddystoneDevice) {
        return instanceId.equals(eddystoneDevice.getInstanceId());
      }
    };
  }
}
