package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.bluetooth.BluetoothDevice;

import android.util.SparseArray;
import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.FramePayload;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.util.ConversionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@SuppressWarnings("MissingPermission")
final class SecureProfileParser extends Parser<SecureProfile> {

  private static final byte[] KONTAKT_SECURE_PROFILE_PREFIX = new byte[] { 0x16, 0x6A, (byte) 0xFE };
  private static final int MIN_SECURE_PROFILE_LENGTH = 14;

  // Payload ID -> Resolver
  private final SparseArray<PayloadResolver> resolvers = new SparseArray<>();

  // Mac Address -> Calculated RSSI
  private final Map<String, Integer> rssiCache = new HashMap<>();

  private ParseListener parseListener;

  final PayloadResolver plainDevicePayloadResolver;
  final ShuffledDevicePayloadResolver shuffledDevicePayloadResolver;
  final PayloadResolver telemetryPayloadResolver;

  SecureProfileParser(final ScanContext scanContext, final FutureShufflesCache shufflesCache) {
    super(scanContext);

    // Create & add default resolvers
    plainDevicePayloadResolver = new PlainDevicePayloadResolver();
    shuffledDevicePayloadResolver = new ShuffledDevicePayloadResolver(shufflesCache);
    telemetryPayloadResolver = new TelemetryPayloadResolver();

    addResolver(plainDevicePayloadResolver);
    addResolver(shuffledDevicePayloadResolver);
    addResolver(telemetryPayloadResolver);

    // Add custom payload resolvers
    final List<PayloadResolver> customPayloadResolvers = scanContext.getCustomSecureProfilePayloadResolvers();
    for (final PayloadResolver payloadResolver : customPayloadResolvers) {
      addResolver(payloadResolver);
    }
  }

  void addListener(final ParseListener parseListener) {
    this.parseListener = parseListener;
  }

  public void addResolver(final PayloadResolver resolver) {
    if (resolver != null) {
      resolver.attachListener(resolveListener);
      resolvers.put(resolver.payloadId, resolver);
    }
  }

  boolean isValidSecureProfileFrame(final byte[] scanRecord) {
    return ConversionUtils.doesArrayContainSubset(scanRecord, KONTAKT_SECURE_PROFILE_PREFIX, 4) && scanRecord.length > MIN_SECURE_PROFILE_LENGTH;
  }

  void parseSecureProfile(final BluetoothDevice device, int rssi, final byte[] scanRecord) {
    // Extract frame data
    final Map<FrameDataType, byte[]> frameData = new HashMap<>();
    extractFrameData(scanRecord, frameData);

    // Extract payload ID
    final byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    final byte payloadId = serviceData[2];

    // Check if there is any corresponding payload resolver
    final PayloadResolver resolver = resolvers.get(payloadId);
    if (resolver != null) {
      // Extract device MAC address
      final String address = device.getAddress();

      // Update RSSI for device
      updateRssi(address, rssi);

      // Dispatch payload to resolver and wait for callback
      final FramePayload payload = new FramePayload(address, frameData);
      resolver.resolve(payload);
    }
  }

  private void updateRssi(final String macAddress, final int rssi) {
    final int calculatedRssi = rssiCalculator.calculateRssi(macAddress.hashCode(), rssi);
    rssiCache.put(macAddress, calculatedRssi);
  }

  private final ResolveListener resolveListener = new ResolveListener() {
    @Override
    public void onResolved(SecureProfile profile) {
      // Get address with hashcode
      final String macAddress = profile.getMacAddress();
      final int hashCode = hashCodeBuilder.append(macAddress).build();

      // Check if device already cached
      final SecureProfile cachedProfile = devicesCache.get(hashCode);
      if (cachedProfile != null) {
        // Update if already cached
        profile = cachedProfile.updateWith(profile);
      } else {
        // Insert new device if not cached yet
        devicesCache.put(hashCode, profile);
      }

      // Update RSSI
      final int rssi = rssiCache.get(macAddress);
      profile.setRssi(rssi);

      // Do callback
      parseListener.onParsed(profile);
    }
  };

  void handleLostEvent(ISecureProfile profile) {
    shuffledDevicePayloadResolver.handleLostEvent(profile);
  }

  @Override
  protected void disable() {
    if (isEnabled) {
      isEnabled = false;
    }
    shuffledDevicePayloadResolver.disableShuffleResolver();
  }

}

