package com.kontakt.sdk.android.ble.cache;

import android.content.Context;
import android.util.TimingLogger;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.discovery.ShuffledDevicesResolver;
import com.kontakt.sdk.android.ble.discovery.ShuffledSecureProfileResolver;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.model.SecureProfileUid;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

@SuppressWarnings("WeakerAccess")
public class FutureShufflesCache {

  private static final String PHANTOM_LABEL = "-PHANTOM-";
  private static final String TAG = "FSCache";
  private static final int DEFAULT_RESOLVER_POOL_SIZE = 2;
  private static final int INITIAL_IBEACON_RESOLVER_DELAY = 1;
  private static final int INITIAL_EDDYSTONE_RESOLVER_DELAY = 2;
  private static final int INITIAL_SECURE_PROFILE_RESOLVER_DELAY = 3;
  public static final ResolvedId PHANTOM_ENTRY = ResolvedId.create(PHANTOM_LABEL, PHANTOM_LABEL, null);

  private final Map<String, ResolvedId> internalCache = new ConcurrentHashMap<>();
  private final Set<String> ignored = new ConcurrentSkipListSet<>();
  private final IBeaconIdResolver iBeaconIdResolver;
  private final EddystoneUIDResolver eddystoneUIDResolver;
  private final SecureProfileResolver secureProfileResolver;
  private final Context context;
  private final ScanContext scanContext;
  private ScheduledExecutorService executorService;
  CacheState state = CacheState.INITIALIZING;

  private final List<ShuffledDevicesResolver.ResolveCallback> deviceResolveCallbacks = new ReplacingArrayList<>();
  private final List<ShuffledSecureProfileResolver.ResolveCallback> secureProfileResolveCallbacks = new ReplacingArrayList<>();

  public FutureShufflesCache(Context context, KontaktCloud kontaktCloud, ScanContext scanContext) {
    this.context = checkNotNull(context);
    this.scanContext = checkNotNull(scanContext);
    this.iBeaconIdResolver = new IBeaconIdResolver(this, kontaktCloud);
    this.eddystoneUIDResolver = new EddystoneUIDResolver(this, kontaktCloud);
    this.secureProfileResolver = new SecureProfileResolver(this, kontaktCloud);
    new DeserializerThread("cache-deserializer-thread").start();
  }

  public void addCallback(final ShuffledDevicesResolver.ResolveCallback callback) {
    deviceResolveCallbacks.add(callback);
  }

  public void addCallback(final ShuffledSecureProfileResolver.ResolveCallback callback) {
    secureProfileResolveCallbacks.add(callback);
  }

  public ResolvedId get(ISecureProfile secureProfile) {
    String fakeProfileUID = SecureProfileUid.fromDevice(secureProfile).toString();
    return get(fakeProfileUID);
  }

  public ResolvedId get(RemoteBluetoothDevice origin) {
    switch (origin.getProfile()) {
      case IBEACON:
        IBeaconDevice iBeaconDevice = (IBeaconDevice) origin;
        String fakeId = IBeaconId.fromDevice(iBeaconDevice).toString();
        return get(fakeId);
      case EDDYSTONE:
        IEddystoneDevice eddystoneDevice = (IEddystoneDevice) origin;
        String fakeUID = EddystoneUid.fromDevice(eddystoneDevice).toString();
        return get(fakeUID);
      default:
        throw new IllegalArgumentException("Unsupported device profile: " + origin.getProfile());
    }
  }

  public ResolvedId get(String deviceId) {
    if (internalCache.containsKey(deviceId)) {
      return internalCache.get(deviceId);
    } else if (ignored.contains(deviceId)) {
      return PHANTOM_ENTRY;
    } else {
      return null;
    }
  }

  public void addResolveRequest(RemoteBluetoothDevice device) {
    createResolverRunners();
    switch (device.getProfile()) {
      case IBEACON:
        IBeaconResolveRequest iBeaconRequest = IBeaconResolveRequest.of(device);
        iBeaconIdResolver.addResolveRequest(iBeaconRequest);
        break;
      case EDDYSTONE:
        EddystoneResolveRequest eddystoneRequest = EddystoneResolveRequest.of(device);
        eddystoneUIDResolver.addResolveRequest(eddystoneRequest);
        break;
      default:
        throw new IllegalArgumentException("Unsupported device profile: " + device.getProfile());
    }
  }

  public void addResolveRequest(ISecureProfile secureProfile) {
    createResolverRunners();
    SecureProfileResolveRequest secureProfileRequest = new SecureProfileResolveRequest(secureProfile);
    secureProfileResolver.addResolveRequest(secureProfileRequest);
  }

  void evict(List<String> uniqueIds, DeviceProfile deviceProfile) {
    Iterator<Map.Entry<String, ResolvedId>> cacheIterator = internalCache.entrySet().iterator();
    while (cacheIterator.hasNext()) {
      Map.Entry<String, ResolvedId> entry = cacheIterator.next();
      ResolvedId evictCandidate = entry.getValue();
      if (uniqueIds.contains(evictCandidate.getUniqueId()) && deviceProfile == evictCandidate.getDeviceProfile()) {
        cacheIterator.remove();
      }
    }
  }

  void createResolverRunners() {
    if (executorService == null) {
      executorService = Executors.newScheduledThreadPool(DEFAULT_RESOLVER_POOL_SIZE);
      executorService.scheduleWithFixedDelay(iBeaconIdResolver, INITIAL_IBEACON_RESOLVER_DELAY, scanContext.getResolveInterval(), TimeUnit.SECONDS);
      executorService.scheduleWithFixedDelay(eddystoneUIDResolver, INITIAL_EDDYSTONE_RESOLVER_DELAY, scanContext.getResolveInterval(),
          TimeUnit.SECONDS);
      executorService.scheduleWithFixedDelay(secureProfileResolver, INITIAL_SECURE_PROFILE_RESOLVER_DELAY, scanContext.getResolveInterval(),
          TimeUnit.SECONDS);
    }
  }

  public void finishResolveRunners() {
    if (executorService != null) {
      executorService.shutdownNow();
      executorService = null;
    }
  }

  public CacheState getState() {
    return state;
  }

  public boolean isInitialized() {
    return CacheState.INITIALIZED == state;
  }

  void populate(String deviceId, ResolvedId resolvedId) {
    if (PHANTOM_ENTRY.equals(resolvedId)) {
      ignored.add(deviceId);
    } else {
      internalCache.put(deviceId, resolvedId);
    }
  }

  void notifyListeners(RemoteBluetoothDevice device) {
    for (ShuffledDevicesResolver.ResolveCallback callback : deviceResolveCallbacks) {
      callback.onResolved(device);
    }
  }

  void notifyListeners(ISecureProfile secureProfile) {
    for (ShuffledSecureProfileResolver.ResolveCallback callback : secureProfileResolveCallbacks) {
      callback.onResolved(secureProfile);
    }
  }

  synchronized void serialize() {
    try {
      TimingLogger timings = new TimingLogger(TAG, "Serialization");

      FileOutputStream fos = context.openFileOutput(scanContext.getCacheFileName(), Context.MODE_PRIVATE);
      BufferedOutputStream bos = new BufferedOutputStream(fos);
      ObjectOutputStream os = new ObjectOutputStream(fos);

      os.writeObject(internalCache);

      os.close();
      bos.close();
      fos.close();

      Logger.d(TAG + " Cached size: " + internalCache.size());

      timings.addSplit("Save file");
      timings.dumpToLog();
    } catch (IOException e) {
      Logger.e(TAG + " Error when try to serialize cache: ", e);
    }
  }

  void deserialize() {
    try {
      TimingLogger timings = new TimingLogger(TAG, "Deserialization");

      FileInputStream fis = context.openFileInput(scanContext.getCacheFileName());
      BufferedInputStream bis = new BufferedInputStream(fis);
      ObjectInputStream ois = new ObjectInputStream(bis);

      ConcurrentHashMap<String, ResolvedId> cached = (ConcurrentHashMap<String, ResolvedId>) ois.readObject();

      ois.close();
      bis.close();
      fis.close();

      Logger.d(TAG + " Cached size: " + cached.size());
      internalCache.putAll(cached);

      timings.addSplit("Read file");
      timings.dumpToLog();
    } catch (FileNotFoundException e) {
      Logger.d(TAG + " Cache file not found: " + e.getMessage());
    } catch (IOException | ClassNotFoundException e) {
      Logger.e(TAG + " Error when try to deserialize cache: ", e);
    }
  }

  public void clear() {
    finishResolveRunners();
    internalCache.clear();
    clearBuffers();
    File cache = new File(context.getFilesDir(), scanContext.getCacheFileName());
    if (!cache.exists()) {
      return;
    }
    String deleted = cache.delete() ? "Success" : "Failure";
    Logger.d(TAG + "Deleting cache file... " + deleted);
  }

  public synchronized void clearBuffers() {
    ignored.clear();
    iBeaconIdResolver.clear();
    eddystoneUIDResolver.clear();
    secureProfileResolver.clear();
    Logger.d(TAG + "Cleared internal buffers");
  }

  public void markIgnored(RemoteBluetoothDevice beacon) {
    switch (beacon.getProfile()) {
      case IBEACON:
        iBeaconIdResolver.markIgnored(beacon);
        break;
      case EDDYSTONE:
        eddystoneUIDResolver.markIgnored(beacon);
        break;
      default:
        throw new IllegalArgumentException("Unsupported device profile: " + beacon.getProfile());
    }
  }

  public void markIgnored(ISecureProfile secureProfile) {
    secureProfileResolver.markIgnored(secureProfile);
  }

  private enum CacheState {
    INITIALIZING,
    INITIALIZED
  }

  private class DeserializerThread extends Thread {

    DeserializerThread(String name) {
      super(name);
    }

    @Override
    public void run() {
      deserialize();
      FutureShufflesCache.this.state = CacheState.INITIALIZED;
    }
  }
}
