package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * Utility model class that represents firmware revision with possibility to compare versions.
 */
public class FirmwareVersion implements Comparable<FirmwareVersion>, Parcelable {

  public static final Parcelable.Creator<FirmwareVersion> CREATOR = new Parcelable.Creator<FirmwareVersion>() {
    @Override
    public FirmwareVersion createFromParcel(Parcel source) {
      return new FirmwareVersion(source);
    }

    @Override
    public FirmwareVersion[] newArray(int size) {
      return new FirmwareVersion[size];
    }
  };

  private final String version;

  public FirmwareVersion(final String version) {
    if (version == null) {
      throw new IllegalArgumentException("Version can not be null");
    }
    if (!version.matches("[0-9]+(\\.[0-9]+)*"))  {
      throw new IllegalArgumentException("Invalid version format");
    }
    this.version = version;
  }

  protected FirmwareVersion(Parcel in) {
    this.version = in.readString();
  }

  public final String get() {
    return version;
  }

  public boolean isGreaterThan(final FirmwareVersion other) {
    return this.compareTo(other) == 1;
  }

  public boolean isGreaterOrEqualTo(final FirmwareVersion other) {
    return isGreaterThan(other) || isEqualTo(other);
  }

  public boolean isLowerThan(final FirmwareVersion other) {
    return this.compareTo(other) == -1;
  }

  public boolean isLowerOrEqualTo(final FirmwareVersion other) {
    return isLowerThan(other) || isEqualTo(other);
  }

  public boolean isEqualTo(final FirmwareVersion other) {
    return this.compareTo(other) == 0;
  }

  @Override
  public String toString() {
    return version;
  }

  @Override
  public int compareTo(final FirmwareVersion that) {
    final String[] thisParts = this.get().split("\\.");
    final String[] thatParts = that.get().split("\\.");
    int length = Math.max(thisParts.length, thatParts.length);
    for(int i = 0; i < length; i++) {
      int thisPart = i < thisParts.length ?
          Integer.parseInt(thisParts[i]) : 0;
      int thatPart = i < thatParts.length ?
          Integer.parseInt(thatParts[i]) : 0;
      if(thisPart < thatPart)
        return -1;
      if(thisPart > thatPart)
        return 1;
    }
    return 0;
  }

  @Override
  public boolean equals(final Object that) {
    if (this == that)
      return true;
    if (that == null)
      return false;
    if (this.getClass() != that.getClass())
      return false;
    return this.compareTo((FirmwareVersion) that) == 0;
  }

  @Override
  public int hashCode() {
    return version.hashCode();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.version);
  }

}
