package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.List;

/**
 * Aggregated set of {@link Event} objects.
 */
public class EventPacket implements Parcelable {

  public static final Parcelable.Creator<EventPacket> CREATOR = new Parcelable.Creator<EventPacket>() {
    @Override
    public EventPacket createFromParcel(Parcel source) {
      return new EventPacket(source);
    }

    @Override
    public EventPacket[] newArray(int size) {
      return new EventPacket[size];
    }
  };

  private String sourceId;
  private SourceType sourceType;
  private long timestamp;
  private List<Event> events = new ArrayList<>();

  public static EventPacket.Builder builder() {
    return new Builder();
  }

  /**
   * Creates new event packet from list of {@link Event} objects and a timestamp.
   *
   * @param events the list of events.
   * @param timestamp the timestamp.
   * @return new event packet instance.
   */
  public static EventPacket from(final List<Event> events, long timestamp) {
    SDKPreconditions.checkNotNull(events, "events cannot be null");
    return new EventPacket(events, timestamp);
  }

  private EventPacket(List<Event> events, long timestamp) {
    this.events = events;
    this.timestamp = timestamp / 1000;
  }

  EventPacket(Builder builder) {
    this.sourceId = builder.sourceId;
    this.sourceType = builder.sourceType;
    this.timestamp = builder.timestamp / 1000;
    this.events = builder.events;
  }

  protected EventPacket(Parcel in) {
    this.sourceId = in.readString();
    int tmpSourceType = in.readInt();
    this.sourceType = tmpSourceType == -1 ? null : SourceType.values()[tmpSourceType];
    this.timestamp = in.readLong();
    this.events = in.createTypedArrayList(Event.CREATOR);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.sourceId);
    dest.writeInt(this.sourceType == null ? -1 : this.sourceType.ordinal());
    dest.writeLong(this.timestamp);
    dest.writeTypedList(this.events);
  }

  public String getSourceId() {
    return sourceId;
  }

  public SourceType getSourceType() {
    return sourceType;
  }

  public long getTimestamp() {
    return timestamp;
  }

  public List<Event> getEvents() {
    return events;
  }

  /**
   * Builder class that is used to build {@link EventPacket} instances from values configured by the setters.
   */
  public static class Builder {
    String sourceId;
    SourceType sourceType;
    long timestamp;
    List<Event> events = new ArrayList<>();

    public Builder sourceId(final String sourceId) {
      this.sourceId = sourceId;
      return this;
    }

    public Builder sourceType(final SourceType sourceType) {
      this.sourceType = sourceType;
      return this;
    }

    public Builder timestamp(final long timestamp) {
      this.timestamp = timestamp;
      return this;
    }

    public Builder events(final List<Event> events) {
      this.events = events;
      return this;
    }

    public EventPacket build() {
      return new EventPacket(this);
    }
  }
}
