package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.List;

/**
 * Represents API response with resolved Eddystone UIDs (from shuffled) and set of future shuffles.
 */
public class EddystoneFutureUID implements IFutureId<EddystoneUid> {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<EddystoneFutureUID> CREATOR = new Creator<EddystoneFutureUID>() {
    @Override
    public EddystoneFutureUID createFromParcel(Parcel source) {
      return new EddystoneFutureUID(source);
    }

    @Override
    public EddystoneFutureUID[] newArray(int size) {
      return new EddystoneFutureUID[size];
    }
  };

  private String uniqueId;
  private EddystoneUid queriedBy;
  private EddystoneUid resolved;
  private List<EddystoneUid> futureIds;

  public static Builder builder() {
    return new Builder();
  }

  EddystoneFutureUID(Builder builder) {
    this.uniqueId = builder.uniqueId;
    this.queriedBy = builder.queriedBy;
    this.resolved = builder.resolved;
    this.futureIds = builder.futureIds;
  }

  protected EddystoneFutureUID(Parcel in) {
    this.uniqueId = in.readString();
    this.queriedBy = in.readParcelable(EddystoneUid.class.getClassLoader());
    this.resolved = in.readParcelable(EddystoneUid.class.getClassLoader());
    this.futureIds = in.createTypedArrayList(EddystoneUid.CREATOR);
  }

  public EddystoneUid getQueriedBy() {
    return queriedBy;
  }

  public EddystoneUid getResolved() {
    return resolved;
  }

  public List<EddystoneUid> getFutureIds() {
    return futureIds;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof EddystoneFutureUID)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    EddystoneFutureUID other = (EddystoneFutureUID) o;
    return SDKEqualsBuilder.start()
        .equals(resolved, other.resolved)
        .equals(queriedBy, other.queriedBy)
        .equals(futureIds, other.futureIds)
        .equals(uniqueId, other.uniqueId)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(resolved).append(queriedBy).append(futureIds).append(uniqueId).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.uniqueId);
    dest.writeParcelable(this.queriedBy, flags);
    dest.writeParcelable(this.resolved, flags);
    dest.writeTypedList(futureIds);
  }

  /**
   * Builder class that is used to build {@link EddystoneFutureUID} instances from values configured by the setters.
   */
  public static class Builder {
    String uniqueId;
    EddystoneUid queriedBy;
    EddystoneUid resolved;
    List<EddystoneUid> futureIds = new ArrayList<>();

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder queriedBy(final EddystoneUid queriedBy) {
      this.queriedBy = queriedBy;
      return this;
    }

    public Builder resolved(final EddystoneUid resolved) {
      this.resolved = resolved;
      return this;
    }

    public Builder futureIds(final List<EddystoneUid> futureIds) {
      SDKPreconditions.checkNotNull(futureIds, "future IDs cannot be null");
      this.futureIds.clear();
      this.futureIds.addAll(futureIds);
      return this;
    }

    public EddystoneFutureUID build() {
      return new EddystoneFutureUID(this);
    }
  }
}
