package com.kontakt.sdk.android.common;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import com.kontakt.sdk.android.common.log.LogLevel;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.UUID;

/**
 * Entry point for Kontakt.io SDK initialization.
 */
public final class KontaktSDK {

  /**
   * The constant DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID. This proximity UUID value
   * is preset as the default for every iBeacon belonging to Kontakt.io
   */
  public static final UUID DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID = UUID.fromString("f7826da6-4fa2-4e98-8024-bc5b71e0893e");

  /**
   * The constant DEFAULT_KONTAKT_NAMESPACE_ID. This namespace ID value
   * is preset as the default for every Eddystone Beacon belonging to Kontakt.io
   */
  public static final String DEFAULT_KONTAKT_NAMESPACE_ID = "f7826da6bc5b71e0893e";

  private static KontaktSDK SINGLETON;
  private final String apiKey;

  private KontaktSDK(final String apiKey) {
    this.apiKey = apiKey;
    Logger.reset();
  }

  /**
   * Initializes SDK with given api key
   *
   * @param apiKey the api key
   * @return KontaktSDK
   */
  public static synchronized KontaktSDK initialize(final String apiKey) {
    SINGLETON = new KontaktSDK(apiKey);

    return SINGLETON;
  }

  /**
   * Initializes SDK from Context by reading api key from AndroidManifest
   * <p>
   * throws {@link IllegalStateException} if package name could not been found
   * <br>
   * throws {@link IllegalArgumentException} if api key has not been found in AndroidManifest
   *
   * @param context the context
   * @return KontaktSDK
   */
  public static synchronized KontaktSDK initialize(final Context context) {
    SDKPreconditions.checkNotNull(context, "Context can't be null");

    ApplicationInfo appInfo;
    try {
      appInfo = context.getApplicationContext().getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
    } catch (PackageManager.NameNotFoundException e) {
      throw new IllegalStateException("Could not find application package name");
    }

    SDKPreconditions.checkArgument(appInfo != null,
        "Application info not found in the manifest. Please check your AndroidManifest.xml");
    SDKPreconditions.checkArgument(appInfo.metaData != null,
        "Meta data not found in the manifest. Please provide meta-data in the AndroidManifest.xml");

    final String apiKey = appInfo.metaData.getString("kontakt.io.API_KEY");
    SDKPreconditions.checkArgument(!TextUtils.isEmpty(apiKey), "API key is not present. Please provide the key in the AndroidManifest.xml");

    return initialize(apiKey);
  }

  /**
   * @return True if Kontakt SDK has been successfully initialized. False otherwise.
   */
  public static synchronized boolean isInitialized() {
    return SINGLETON != null;
  }

  /**
   * @return {@link KontaktSDK} singleton instance.
   */
  public static synchronized KontaktSDK getInstance() {
    SDKPreconditions.checkNotNull(SINGLETON, "Kontakt.io SDK has not been initialized. Please, invoke initialize() method first");
    return SINGLETON;
  }

  /**
   * Resets current Kontakt SDK instance.
   */
  public static synchronized void reset() {
    SINGLETON = null;
  }

  /**
   * @return Currently used API KEY
   */
  public final String getApiKey() {
    return apiKey;
  }

  /**
   * Enables/disables Kontakt SDK logging.
   *
   * @param state true or false.
   * @return {@link KontaktSDK} instance.
   */
  public final KontaktSDK setDebugLoggingEnabled(final boolean state) {
    Logger.setDebugLoggingEnabled(state);
    return this;
  }

  /**
   * Enables/disables given logging level.
   *
   * @param level {@link LogLevel} log level type.
   * @param state enabled or disabled.
   * @return {@link KontaktSDK} instance.
   */
  public final KontaktSDK setLogLevelEnabled(final LogLevel level, final boolean state) {
    Logger.setLogLevelEnabled(level, state);
    return this;
  }

  /**
   * Enables/disables Kontakt SDK Crashlytics logging.
   *
   * @param state true or false.
   * @return {@link KontaktSDK} instance.
   */
  public final KontaktSDK setCrashlyticsLoggingEnabled(final boolean state) {
    Logger.setCrashlyticsLoggingEnabled(state);
    return this;
  }
}
