package com.kontakt.sdk.android.common;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import static com.kontakt.sdk.android.common.util.ArrayUtils.concat;
import static java.util.Arrays.asList;

/**
 * Representation of available firmware revisions
 */
public final class FirmwareRevisions {

  /**
   * Set of Beacon Pro firmwares revisions
   */
  private static final String[] BEACON_PRO_VERSIONS = new String[] { "1.1", "1.2", "1.3", "1.4", "1.5", "1.6", "1.7", "1.8", "1.9", "1.10" };

  /**
   * Set of hardware Gateway firmwares revisions
   */
  private static final String[] GATEWAY_VERSIONS = new String[]{
      "1.0", "1.0.5", "1.0.6", "1.0.7", "1.10", "1.1.2", "1.1.3", "1.1.4", "1.1.5",
      "1.2", "1.2.1", "1.2.2", "1.2.3", "1.2.4", "1.2.5",
      "1.3", "1.3.1", "1.3.2", "1.3.3", "1.3.4",
      "1.4",
      "1.5", "1.5.1", "1.5.2", "1.5.3", "1.5.4", "1.5.5"
  };

  /**
   * Set of Beacon Pro firmwares revisions that support RTC power saving and reading light sensor
   */
  public static final Set<String> BEACON_PRO_RTC_LIGHTSENSOR_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "1.7", "1.8", "1.9", "1.10" })));

  /**
   * Set of Beacon Pro firmwares revisions that support Kontakt TLM packet
   */
  public static final Set<String> BEACON_PRO_KONTAKT_TLM_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "1.10" })));

  /**
   * Set of firmware revisions of beacon devices with single writable characteristic available
   */
  public static final Set<String> SECURE_MODE_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(concat(concat(new String[]{"4.0", "4.1", "4.2"}, BEACON_PRO_VERSIONS), GATEWAY_VERSIONS))));

  /**
   * Set of firmware revisions of Beacon Pro devices.
   */
  public static final Set<String> BEACON_PRO_FIRMWARE_VERSIONS = Collections.unmodifiableSet(new HashSet<>(asList(BEACON_PRO_VERSIONS)));

  /**
   * Set of firmware revisions of beacon devices which support interleaving.
   */
  public static final Set<String> INTERLEAVING_PACKETS_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(concat(new String[]{"4.1", "4.2"}, BEACON_PRO_VERSIONS))));

  /**
   * Checks if provided firmware version is eligible for multiple packets.
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isInterleavingAvailable(String firmwareVersion) {
    return INTERLEAVING_PACKETS_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware version is eligible for multiple packets.
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isSecure(String firmwareVersion) {
    return SECURE_MODE_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware version is a beacon pro version.
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isPro(String firmwareVersion) {
    return BEACON_PRO_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }
}
