package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.managers.CreateManagerRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.managers.DeleteManagerRequestManager;
import com.kontakt.sdk.android.cloud.api.executor.managers.ManagerRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.managers.ManagersRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.managers.UpdateManagerRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ManagersService;
import com.kontakt.sdk.android.common.model.Manager;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * An abstract representation of Managers API endpoint. Provides executors that can be
 * used to handle all manager-related requests.
 */
public class ManagersApi {

  private final ManagersService managersService;

  /**
   * Constructs Managers API endpoint initialized with corresponding service class.
   *
   * @param managersService the managers API service.
   */
  public ManagersApi(final ManagersService managersService) {
    this.managersService = managersService;
  }

  /**
   * Provides executor that can be used to obtain single manager by ID.
   *
   * @param id the manager's unique identifier.
   * @return the request executor.
   */
  public ManagerRequestExecutor fetch(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new ManagerRequestExecutor(managersService, id);
  }

  /**
   * Provides executor that can be used to obtain managers.
   *
   * @return the request executor.
   */
  public ManagersRequestExecutor fetch() {
    return new ManagersRequestExecutor(managersService);
  }

  /**
   * Provides executor that can be used to create new manager.
   *
   * @param manager the manager.
   * @return teh request executor.
   */
  public CreateManagerRequestExecutor create(final Manager manager) {
    SDKPreconditions.checkNotNull(manager, "manager cannot be null");
    return new CreateManagerRequestExecutor(managersService, manager);
  }

  /**
   * Provides executor that can be used to update a manager.
   *
   * @param managerId the manager's unique identifier.
   * @return the request executor.
   */
  public UpdateManagerRequestExecutor update(final UUID managerId) {
    SDKPreconditions.checkNotNull(managerId, "manager ID cannot be null");
    return new UpdateManagerRequestExecutor(managersService, managerId);
  }

  /**
   * Provides executor that can be used to delete a manager.
   *
   * @param managerId the manager's unique identifier.
   * @return the request executor.
   */
  public DeleteManagerRequestManager delete(final UUID managerId) {
    SDKPreconditions.checkNotNull(managerId, "manager ID cannot be null");
    return new DeleteManagerRequestManager(managersService, managerId);
  }
}
