package com.kontakt.sdk.android.ble.security.parser;

import com.kontakt.sdk.android.ble.security.ResponseCode;
import com.kontakt.sdk.android.ble.security.exception.InvalidConfigException;
import com.kontakt.sdk.android.ble.security.property.AbstractProperty;
import com.kontakt.sdk.android.ble.security.property.ByteArrayProperty;
import com.kontakt.sdk.android.ble.security.property.Int8Property;
import com.kontakt.sdk.android.ble.security.property.PropertyID;
import com.kontakt.sdk.android.ble.security.property.StringProperty;
import com.kontakt.sdk.android.ble.security.property.TerminatingProperty;
import com.kontakt.sdk.android.ble.security.property.UInt16Property;
import com.kontakt.sdk.android.ble.security.property.UInt32Property;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

public class DataResponseParser extends SimpleResponseParser {

  private static final int PID_LENGTH = 2;

  public static DataResponseParser parse(byte[] response) throws InvalidConfigException {
    return new DataResponseParser(response, null);
  }

  public static DataResponseParser parse(byte[] response, String password) throws InvalidConfigException {
    return new DataResponseParser(response, password);
  }

  private DataResponseParser(byte[] response, String password) throws InvalidConfigException {
    super(response, password);
  }

  public List<AbstractProperty<?>> getProperties() {
    if (result != ResponseCode.OK) {
      throw new IllegalStateException("Cannot get properties when response was not OK");
    }
    ByteBuffer buffer = ByteBuffer.wrap(data);
    List<AbstractProperty<?>> properties = new ArrayList<>();
    PropertyID currentPropertyId;

    do {
      int length = buffer.get();
      currentPropertyId = PropertyID.of(Short.reverseBytes(buffer.getShort()));
      byte[] value = new byte[length - PID_LENGTH];
      buffer.get(value);
      if (currentPropertyId == null) {
        continue;
      }
      properties.add(createProperty(currentPropertyId, value));
    }
    while (currentPropertyId != PropertyID.TERMINATING);

    buffer.clear();
    return properties;
  }

  private AbstractProperty createProperty(PropertyID currentPropertyId, byte[] value) {
    switch (currentPropertyId.getDataType()) {
      case BYTE_ARRAY:
        return new ByteArrayProperty(currentPropertyId, value);
      case INT8:
      case UINT8:
        return new Int8Property(currentPropertyId, value);
      case NONE:
        return new TerminatingProperty(currentPropertyId, value);
      case STRING:
        return new StringProperty(currentPropertyId, value);
      case UINT16:
        return new UInt16Property(currentPropertyId, value);
      case UINT32:
        return new UInt32Property(currentPropertyId, value);
      default:
        throw new RuntimeException("Invalid property type");
    }
  }
}
