package com.kontakt.sdk.android.ble.manager.internal;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

@SuppressWarnings("WeakerAccess")
class ShuffledSpacesManager {

  final List<SpacesResolver> resolvers = new ArrayList<>();
  OnSpacesResolvedListener onSpacesResolvedListener = OnSpacesResolvedListener.NOOP_LISTENER;
  private ScanContext scanContext;

  ShuffledSpacesManager(KontaktCloud cloud) {
    checkNotNull(cloud);
    resolvers.add(new SpacesResolver.RegionsResolver(cloud));
    resolvers.add(new SpacesResolver.NamespacesResolver(cloud));
  }

  //Used in tests
  ShuffledSpacesManager(KontaktCloud cloud, SpacesResolver... resolvers) {
    checkNotNull(cloud);
    Collections.addAll(this.resolvers, resolvers);
  }

  public void resolve(ScanContext scanContext, OnSpacesResolvedListener onSpacesResolvedListener) {
    this.scanContext = checkNotNull(scanContext);
    this.onSpacesResolvedListener = checkNotNull(onSpacesResolvedListener);
    for (SpacesResolver resolver : resolvers) {
      resolver.resolve(scanContext, resolvingStatusListener);
    }
  }

  void onDestroy() {
    clearCache();
    onSpacesResolvedListener = OnSpacesResolvedListener.NOOP_LISTENER;
  }

  void clearCache() {
    for (SpacesResolver resolver : resolvers) {
      resolver.clearCache();
    }
  }

  ScanContext createScanContext(Collection<IBeaconRegion> resolvedRegions, Collection<IEddystoneNamespace> resolvedNamespaces) {
    ScanContext.Builder newContext = new ScanContext.Builder(scanContext);
    if (!resolvedRegions.isEmpty()) {
      newContext.iBeaconRegions(resolvedRegions);
    }
    if (!resolvedNamespaces.isEmpty()) {
      newContext.eddystoneNamespaces(resolvedNamespaces);
    }
    return newContext.build();
  }

  void resetFinishedStatus() {
    for (SpacesResolver resolver : resolvers) {
      resolver.resetFinishedStatus();
    }
  }

  boolean allResolversAreFinishedSuccessfully() {
    for (SpacesResolver resolver : resolvers) {
      if (!resolver.isFinishedSuccessfully()) {
        return false;
      }
    }
    return true;
  }

  private final SpacesResolver.ResolvingStatusListener resolvingStatusListener = new SpacesResolver.ResolvingStatusListener() {
    @Override
    public void onSuccess() {
      if (allResolversAreFinishedSuccessfully()) {
        Collection<IBeaconRegion> regions = resolvers.get(0).getResolvedSpaces();
        Collection<IEddystoneNamespace> namespaces = resolvers.get(1).getResolvedSpaces();
        resetFinishedStatus();
        onSpacesResolvedListener.onSpacesResolved(createScanContext(regions, namespaces));
      }
    }

    @Override
    public void onError(String message) {
      onSpacesResolvedListener.onError(new ScanError(message));
    }
  };

  interface OnSpacesResolvedListener {
    void onSpacesResolved(ScanContext scanContext);

    void onError(ScanError error);

    OnSpacesResolvedListener NOOP_LISTENER = new OnSpacesResolvedListener() {
      @Override
      public void onSpacesResolved(ScanContext scanContext) {
        //No operation
      }

      @Override
      public void onError(ScanError error) {
        //No operation
      }
    };
  }
}
