package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SecureProfileUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class SecureProfileEvent implements BluetoothDeviceEvent {

  public static final Creator<SecureProfileEvent> CREATOR = new Creator<SecureProfileEvent>() {
    @Override
    public SecureProfileEvent createFromParcel(Parcel source) {
      Bundle bundle = source.readBundle(getClass().getClassLoader());
      EventType eventType = (EventType) bundle.getSerializable(PARCELABLE_EVENT_TYPE);
      ArrayList<ISecureProfile> deviceList = bundle.getParcelableArrayList(PARCELABLE_DEVICE_LIST);
      long timestamp = bundle.getLong(PARCELABLE_TIMESTAMP);

      return new SecureProfileEvent(timestamp, eventType, deviceList);
    }

    @Override
    public SecureProfileEvent[] newArray(int size) {
      return new SecureProfileEvent[size];
    }
  };

  private static final String PARCELABLE_EVENT_TYPE = "kontakt_event_type";
  private static final String PARCELABLE_DEVICE_LIST = "remote_device_list";
  private static final String PARCELABLE_TIMESTAMP = "timestamp";

  private final long timestamp;
  private final EventType eventType;
  private final List<ISecureProfile> devices;

  public static SecureProfileEvent of(SecureProfileEvent event, List<ISecureProfile> devices) {
    return new SecureProfileEvent(event.getTimestamp(), event.getEventType(), devices);
  }

  static SecureProfileEvent createNewDiscovered(ISecureProfile secureProfile, long timestamp) {
    return new SecureProfileEvent(timestamp, EventType.DEVICE_DISCOVERED, Collections.singletonList(secureProfile));
  }

  static SecureProfileEvent createNewUpdated(List<ISecureProfile> secureProfiles, long timestamp) {
    return new SecureProfileEvent(timestamp, EventType.DEVICES_UPDATE, Collections.unmodifiableList(secureProfiles));
  }

  static SecureProfileEvent createNewLost(ISecureProfile secureProfile, long timestamp) {
    return new SecureProfileEvent(timestamp, EventType.DEVICE_LOST, Collections.singletonList(secureProfile));
  }

  SecureProfileEvent(long timestamp, EventType eventType, List<ISecureProfile> devices) {
    this.timestamp = timestamp;
    this.eventType = eventType;
    this.devices = devices;
  }

  @Override
  public long getTimestamp() {
    return timestamp;
  }

  @Override
  public EventType getEventType() {
    return eventType;
  }

  @Override
  public DeviceProfile getDeviceProfile() {
    return DeviceProfile.KONTAKT_SECURE;
  }

  public List<ISecureProfile> getSecureProfileDevices() {
    return devices;
  }

  @Override
  public List<? extends RemoteBluetoothDevice> getDeviceList() {
    return SecureProfileUtils.asRemoteBluetoothDevices(devices);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putSerializable(PARCELABLE_EVENT_TYPE, eventType);
    bundle.putParcelableArrayList(PARCELABLE_DEVICE_LIST, (ArrayList<? extends Parcelable>) devices);
    bundle.putLong(PARCELABLE_TIMESTAMP, timestamp);

    dest.writeBundle(bundle);
  }
}
