package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Object representing BeaconRegion
 * Proximity UUID is required for setting region
 * <p>
 * If you want to scan only for Proximity UUID, you can set major to {@link BeaconRegion#ANY_MAJOR} and minor to {@link BeaconRegion#ANY_MINOR}
 *
 * <p>
 * Example usage:
 * <pre>
 *  BeaconRegion region = new BeaconRegion.Builder()
 *    .identifier("The Office")
 *    .proximity("f7826da6-4fa2-4e98-8024-bc5b71e0893e")
 *    .major(12837)
 *    .build();
 * </pre>
 */
public class BeaconRegion implements IBeaconRegion {

  public static final Creator<BeaconRegion> CREATOR = new Creator<BeaconRegion>() {
    @Override
    public BeaconRegion createFromParcel(Parcel source) {
      Bundle bundle = source.readBundle(getClass().getClassLoader());

      return new Builder()
          .proximity(((UUID) bundle.getSerializable(Constants.Region.UUID)))
          .secureProximity(((UUID) bundle.getSerializable(Constants.Region.SECURE_UUID)))
          .major(bundle.getInt(Constants.Region.MAJOR))
          .minor(bundle.getInt(Constants.Region.MINOR))
          .identifier(bundle.getString(Constants.Region.IDENTIFIER))
          .build();
    }

    @Override
    public BeaconRegion[] newArray(int size) {
      return new BeaconRegion[size];
    }
  };

  /**
   * Value marking than any major is acceptable
   */
  public static final int ANY_MAJOR = -1;
  /**
   * Value marking than any minor is acceptable
   */
  public static final int ANY_MINOR = -1;

  /**
   * Default region for scan. Accepts all beacons discovered during scan
   */
  public static final BeaconRegion EVERYWHERE = new BeaconRegion.Builder()
      .identifier("Everywhere")
      .proximity(KontaktSDK.DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID)
      .secureProximity(null)
      .major(ANY_MAJOR)
      .minor(ANY_MINOR)
      .build();

  private final UUID proximity;
  private final UUID secureProximity;
  private final int major;
  private final int minor;
  private final String identifier;

  public static Builder builder() {
    return new Builder();
  }

  BeaconRegion(Builder builder) {
    checkNotNull(builder.identifier, "Identifier cannot be null");
    SDKPreconditions.checkArgument(builder.proximity != null || builder.secureProximity != null, "Proximity is not set");

    this.proximity = builder.proximity;
    this.secureProximity = builder.secureProximity;
    this.major = builder.major;
    this.minor = builder.minor;
    this.identifier = builder.identifier;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    BeaconRegion that = (BeaconRegion) o;

    if (major != that.major) return false;
    if (minor != that.minor) return false;
    if (proximity != null ? !proximity.equals(that.proximity) : that.proximity != null) return false;
    if (secureProximity != null ? !secureProximity.equals(that.secureProximity) : that.secureProximity != null) return false;
    return identifier.equals(that.identifier);
  }

  @Override
  public int hashCode() {
    int result = proximity != null ? proximity.hashCode() : 0;
    result = 31 * result + (secureProximity != null ? secureProximity.hashCode() : 0);
    result = 31 * result + major;
    result = 31 * result + minor;
    result = 31 * result + identifier.hashCode();
    return result;
  }

  @Override
  public String toString() {
    StringBuilder region = new StringBuilder();
    region.append("[");
    if (proximity != null) {
      region.append(proximity);
    } else if (secureProximity != null) {
      region.append(secureProximity);
    }
    region.append(", ");

    if (major == ANY_MAJOR) {
      region.append("ANY_MAJOR");
    } else {
      region.append(major);
    }
    region.append(", ");

    if (minor == ANY_MINOR) {
      region.append("ANY_MINOR");
    } else {
      region.append(minor);
    }
    region.append("]");

    return region.toString();
  }

  @Override
  public UUID getProximity() {
    return proximity;
  }

  @Override
  public UUID getSecureProximity() {
    return secureProximity;
  }

  @Override
  public int getMajor() {
    return major;
  }

  @Override
  public int getMinor() {
    return minor;
  }

  @Override
  public String getIdentifier() {
    return identifier;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putSerializable(Constants.Region.UUID, proximity);
    bundle.putSerializable(Constants.Region.SECURE_UUID, secureProximity);
    bundle.putInt(Constants.Region.MAJOR, major);
    bundle.putInt(Constants.Region.MINOR, minor);
    bundle.putString(Constants.Region.IDENTIFIER, identifier);
    dest.writeBundle(bundle);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  public static class Builder {

    UUID proximity;
    UUID secureProximity;
    int major = ANY_MAJOR;
    int minor = ANY_MINOR;
    String identifier;

    /**
     * Sets proximity uuid. Use this option if you use beacons that are not shuffled.
     *
     * @param proximity region proximity
     * @return the builder
     */
    public Builder proximity(final UUID proximity) {
      this.proximity = proximity;
      return this;
    }

    /**
     * Sets secure proximity uuid. Use this option only if you are using shuffled Beacons PRO.
     *
     * @param secureProximity secure region proximity
     * @return the builder
     */
    public Builder secureProximity(final UUID secureProximity) {
      this.secureProximity = secureProximity;
      return this;
    }

    /**
     * Sets major
     *
     * @param major region major
     * @return the builder
     */
    public Builder major(int major) {
      this.major = major;
      return this;
    }

    /**
     * Sets minor
     *
     * @param minor region minor
     * @return the builder
     */
    public Builder minor(int minor) {
      this.minor = minor;
      return this;
    }

    /**
     * Sets region identifier
     *
     * @param identifier identifier
     * @return the builder
     */
    public Builder identifier(String identifier) {
      this.identifier = identifier;
      return this;
    }

    /**
     * Builds BeaconRegion
     *
     * @return region
     */
    public BeaconRegion build() {
      return new BeaconRegion(this);
    }
  }
}
