package com.kontakt.sdk.android.ble.configuration;

import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;

import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

final class ScanContextValidator {

  private ScanContextValidator() {
  }

  private static final int MIN_RESOLVE_INTERVAL = 0;
  private static final int MIN_MONITORING_SYNC_INTERVAL = 0;
  private static final int MAX_FILTERS_SIZE = 10;
  private static final int REGIONS_LIMIT = 20;
  private static final String FILE_NAME_REGEX = "^[^(^?|*<\":>+\\[\\]\\/'{})]+$";
  private static final Pattern FILE_NAME_MATCHER = Pattern.compile(FILE_NAME_REGEX);

  static void validate(final ActivityCheckConfiguration activityCheckConfiguration) {
    if (activityCheckConfiguration == ActivityCheckConfiguration.MINIMAL || activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
      return;
    }

    long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();
    long minimalInactivityTimeout = ActivityCheckConfiguration.MINIMAL.getInactivityTimeout();
    checkArgument(inactivityTimeout >= minimalInactivityTimeout, "The inactivity timeout must be no shorter than 3 seconds");

    long checkPeriod = activityCheckConfiguration.getCheckPeriod();
    long minimalCheckPeriod = ActivityCheckConfiguration.MINIMAL.getCheckPeriod();
    checkArgument(checkPeriod >= minimalCheckPeriod, "The check period must be no shorter than 1 second");
  }

  static void validate(final ForceScanConfiguration forceScanConfiguration) {
    if (forceScanConfiguration == ForceScanConfiguration.DISABLED) {
      return;
    }
    final ForceScanConfiguration minimalConfig = ForceScanConfiguration.MINIMAL;

    final long minimalActivePeriod = minimalConfig.getForceScanActivePeriod();
    if (forceScanConfiguration.getForceScanActivePeriod() < minimalActivePeriod) {
      throw new IllegalArgumentException(String.format("The Force Scan Active period must not be shorter than %d milliseconds", minimalActivePeriod));
    }

    final long minimalPassivePeriod = minimalConfig.getForceScanPassivePeriod();
    if (forceScanConfiguration.getForceScanPassivePeriod() < minimalPassivePeriod) {
      throw new IllegalArgumentException(String.format("The Force Scan Passive period must not be shorter than %d", minimalPassivePeriod));
    }
  }

  static void validate(final ScanPeriod scanPeriod) {

    final long activePeriod = scanPeriod.getActivePeriod();
    checkArgument(activePeriod >= ScanPeriod.MINIMAL_ACTIVE_SCAN_PERIOD,
        String.format("Active scan period should last no shorter than %d seconds", ScanPeriod.MINIMAL_ACTIVE_SCAN_PERIOD / 1000));

    final long passiveSubPeriod = scanPeriod.getPassivePeriod();

    checkArgument((passiveSubPeriod == 0L) || (passiveSubPeriod >= ScanPeriod.MINIMAL_PASSIVE_SCAN_PERIOD),
        "Unsupported ScanPeriod's passive period");
  }

  static void validateCacheFileName(String name) {
    boolean match = FILE_NAME_MATCHER.matcher(name).matches();
    checkArgument(match, "Incorrect cache file name");
  }

  static void validate(final ActivityCheckConfiguration activityCheckConfiguration, ScanPeriod scanPeriod) {
    if (scanPeriod != ScanPeriod.RANGING) {
      checkArgument(activityCheckConfiguration.getCheckPeriod() < scanPeriod.getActivePeriod(),
          "Activity check period must be shorter than active scan period");
    }
  }

  static void validateResolveShuffleInterval(int intervalSeconds) {
    checkArgument(intervalSeconds > MIN_RESOLVE_INTERVAL, "Resolve interval must be greater than 0!");
  }

  static void validateMonitoringSyncInterval(int intervalSeconds) {
    checkArgument(intervalSeconds > MIN_MONITORING_SYNC_INTERVAL, "Monitoring sync interval must be greater than 0!");
  }

  static void validateDeviceUpdateCallbackInterval(long intervalMillis) {
    checkArgument(intervalMillis > 0, "Interval must be greater than 0");
  }

  static void validateIBeaconRegionsCount(final Collection<IBeaconRegion> iBeaconRegions) {
    checkNotNull(iBeaconRegions, "Regions are null");
    checkArgument(iBeaconRegions.size() <= REGIONS_LIMIT, "You can range beacons within max 20 regions.");
  }

  static void validateIBeaconFiltersCount(List<IBeaconFilter> filters) {
    checkNotNull(filters, "IBeacon filters are null");
    checkState(filters.size() <= MAX_FILTERS_SIZE, "Too many iBeacon filters specified. The limit is " + MAX_FILTERS_SIZE);
  }

  static void validateEddystoneFiltersCount(List<EddystoneFilter> eddystoneFilterList) {
    checkNotNull(eddystoneFilterList, "Eddystone filters are null");
    checkState(eddystoneFilterList.size() <= MAX_FILTERS_SIZE,
        "Too many eddystone filters specified. The limit is " + MAX_FILTERS_SIZE);
  }

  static void validateNamespacesCount(Collection<IEddystoneNamespace> eddystoneNamespaceSet) {
    checkNotNull(eddystoneNamespaceSet, "Eddystone namespaces are null");
    checkArgument(eddystoneNamespaceSet.size() <= REGIONS_LIMIT, "You can range eddystones within max 20 namespaces.");
  }
}