package com.kontakt.sdk.android.ble.configuration;

import java.util.concurrent.TimeUnit;

/**
 * Activity Check configuration describes the time boundary after which the beacon
 * device is interpreted as absent and than is removed from internal cache.
 * <p>
 * The configuration consists of 2 parameters - removal inactivityTimeout
 * and checkPeriod.
 * <p>
 * The former describes time threshold after which
 * beacon device is interpreted as inactive and should be removed.
 * <p>
 * The latter specifies period of search of inactive beacons.
 */
public final class ActivityCheckConfiguration {

  /**
   * Predefined constant disabling device activity check configuration.
   */
  public static final ActivityCheckConfiguration DISABLED = new ActivityCheckConfiguration(-1, -1);

  /**
   * Predefined minimal allowed Beacon Activity Check Configuration.<br><br>
   * The MINIMAL configuration specifies beacon inactivity timeout equal to 3 seconds. Inactivity check will be performed every 1 seconds.<br>
   * This option is very "aggressive" when it comes to reporting lost beacons. It should only be used for smartphones with really good BLE
   * hardware.<br>
   * In majority of usecases {@link ActivityCheckConfiguration#DEFAULT} is the best option.
   */
  public static final ActivityCheckConfiguration MINIMAL = new ActivityCheckConfiguration(TimeUnit.SECONDS.toMillis(3), TimeUnit.SECONDS.toMillis(1));

  /**
   * Predefined default Beacon Activity Check Configuration.
   * This configuration is set when no different is specified.
   * The DEFAULT configuration specifies beacon inactivity timeout equal to 10 seconds. Inactivity check will be performed every 5 seconds of active
   * scanning.
   * <br><br>
   * This is the best option for majority of use cases.
   */
  public static final ActivityCheckConfiguration DEFAULT =
      new ActivityCheckConfiguration(TimeUnit.SECONDS.toMillis(10), TimeUnit.SECONDS.toMillis(5));

  private final long inactivityTimeout;
  private final long checkPeriod;

  /**
   * Instantiates new immutable Beacon Activity Check Configuration instance
   *
   * @param inactivityTimeout the removal threshold [ms]
   * @param checkPeriod       the removal period [ms]
   * @return ActivityCheckConfiguration instance
   */
  public static ActivityCheckConfiguration create(long inactivityTimeout, long checkPeriod) {
    return new ActivityCheckConfiguration(inactivityTimeout, checkPeriod);
  }

  private ActivityCheckConfiguration(long inactivityTimeout, long checkPeriod) {
    this.inactivityTimeout = inactivityTimeout;
    this.checkPeriod = checkPeriod;
  }

  /**
   * Provides Beacon Inactivity Timeout [ms].
   *
   * @return the beacon inactivity timeout
   */
  public long getInactivityTimeout() {
    return inactivityTimeout;
  }

  /**
   * Provides check period.
   *
   * @return the check period [ms]
   */
  public long getCheckPeriod() {
    return checkPeriod;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    ActivityCheckConfiguration that = (ActivityCheckConfiguration) o;

    if (inactivityTimeout != that.inactivityTimeout) return false;
    return checkPeriod == that.checkPeriod;
  }

  @Override
  public int hashCode() {
    int result = (int) (inactivityTimeout ^ (inactivityTimeout >>> 32));
    result = 31 * result + (int) (checkPeriod ^ (checkPeriod >>> 32));
    return result;
  }
}
