package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.model.SecureProfileFutureUID;
import com.kontakt.sdk.android.common.model.SecureProfileUid;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;

class SecureProfileResolver implements Runnable {

  private static final String TAG = "SecureProfileResolver";
  private static final int REQUEST_UNIT_SIZE = 120;
  private static final int DEFAULT_BUFFER_SIZE = 200;
  private static final DeviceProfile DEVICE_PROFILE = DeviceProfile.KONTAKT_SECURE;
  private final ArrayBlockingQueue<SecureProfileResolveRequest> requestQueue;
  private final FutureShufflesCache cache;
  private final Collection<SecureProfileResolveStrategy> strategies;

  SecureProfileResolver(FutureShufflesCache futureShufflesCache, KontaktCloud kontaktCloud) {
    this.cache = futureShufflesCache;
    this.requestQueue = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
    List<SecureProfileResolveStrategy> resolvers = new LinkedList<>();
    SecureProfileResolveStrategy cacheStrategy = new SecureProfileCacheResolveStrategy(cache);
    SecureProfileResolveStrategy apiStrategy = new SecureProfileApiResolveStrategy(kontaktCloud);
    resolvers.add(cacheStrategy);
    resolvers.add(apiStrategy);
    this.strategies = Collections.unmodifiableCollection(resolvers);
  }

  @Override
  public void run() {
    if (!cache.isInitialized()) {
      Logger.d(TAG + " Cache not initialized yet");
      return;
    }
    List<SecureProfileResolveRequest> requests = new ArrayList<>();
    requestQueue.drainTo(requests, REQUEST_UNIT_SIZE);
    if (requests.isEmpty()) {
      Logger.d(TAG + " Nothing to resolve");
      return;
    }
    try {
      Logger.d(TAG + " Start resolving");

      final Map<SecureProfileUid, SecureProfileResolveRequest> requestsRegister = buildRequestsRegister(requests);
      final List<SecureProfileFutureUID> shuffles = resolveShuffles(requestsRegister);
      final Map<SecureProfileUid, SecureProfileFutureUID> shufflesRegister = buildShufflesRegister(shuffles);

      evictOutdatedCacheEntries(requestsRegister, shufflesRegister);
      handleRequests(requestsRegister, shufflesRegister);

      cache.serialize();
    } catch (Exception e) {
      Throwable cause = e.getCause();
      if (UnknownHostException.class.isInstance(cause) || SocketTimeoutException.class.isInstance(cause)) {
        requestQueue.addAll(requests);
      } else {
        Logger.e(TAG + " Error occurs when try to resolve shuffled device ", e);
      }
    }
  }

  void addResolveRequest(SecureProfileResolveRequest request) {
    if (requestQueue.contains(request)) {
      return;
    }
    try {
      requestQueue.add(request);
    } catch (IllegalStateException fullQueueException) {
      Logger.e("Could not add Secure Profiles to resolve", fullQueueException);
    }
  }

  public void markIgnored(ISecureProfile secureProfile) {
    for (SecureProfileResolveRequest request : requestQueue) {
      if (request.getFakeDevice().equals(secureProfile)) {
        request.setStatus(ResolveRequestStatus.IGNORED);
      }
    }
  }

  public void clear() {
    requestQueue.clear();
  }

  private Map<SecureProfileUid, SecureProfileResolveRequest> buildRequestsRegister(final List<SecureProfileResolveRequest> requests) {
    final Map<SecureProfileUid, SecureProfileResolveRequest> requestsRegister = new HashMap<>();
    for (SecureProfileResolveRequest request : requests) {
      SecureProfileUid secureProfileUid = SecureProfileUid.fromDevice(request.getFakeDevice());
      requestsRegister.put(secureProfileUid, request);
    }
    return requestsRegister;
  }

  private List<SecureProfileFutureUID> resolveShuffles(final Map<SecureProfileUid, SecureProfileResolveRequest> requestsRegister) throws Exception {
    final List<SecureProfileFutureUID> shuffles = new ArrayList<>();
    for (SecureProfileResolveStrategy strategy : strategies) {
      List<SecureProfileFutureUID> resolved = strategy.resolve(requestsRegister);
      shuffles.addAll(resolved);
    }
    return shuffles;
  }

  private Map<SecureProfileUid, SecureProfileFutureUID> buildShufflesRegister(final List<SecureProfileFutureUID> shuffles) {
    final Map<SecureProfileUid, SecureProfileFutureUID> shufflesRegister = new HashMap<>();
    for (SecureProfileFutureUID futureShuffle : shuffles) {
      shufflesRegister.put(futureShuffle.getQueriedBy(), futureShuffle);
    }
    return shufflesRegister;
  }

  private void evictOutdatedCacheEntries(final Map<SecureProfileUid, SecureProfileResolveRequest> requestsRegister,
      final Map<SecureProfileUid, SecureProfileFutureUID> shufflesRegister) {
    final List<String> uniqueIds = new ArrayList<>();
    for (Map.Entry<SecureProfileUid, SecureProfileResolveRequest> entry : requestsRegister.entrySet()) {
      SecureProfileUid queriedBy = entry.getKey();
      SecureProfileResolveRequest request = entry.getValue();
      SecureProfileFutureUID futureUID = shufflesRegister.get(queriedBy);
      if (futureUID == null || ResolverType.CACHE == request.getResolvedBy()) {
        continue;
      }
      // non null shuffles resolved by api
      uniqueIds.add(futureUID.getUniqueId());
    }

    // remove old (OLD_DEVICE_ID -> RESOLVED_ID) entries from cache and
    // replace them with the new ones (NEW_DEVICE_ID -> RESOLVED_ID) in next step (@addNewCacheEntries)
    cache.evict(uniqueIds, DEVICE_PROFILE);
  }

  private void handleRequests(final Map<SecureProfileUid, SecureProfileResolveRequest> requestsRegister,
      final Map<SecureProfileUid, SecureProfileFutureUID> shufflesRegister) {
    for (Map.Entry<SecureProfileUid, SecureProfileResolveRequest> entry : requestsRegister.entrySet()) {
      handleRequest(shufflesRegister, entry);
    }
  }

  private void handleRequest(Map<SecureProfileUid, SecureProfileFutureUID> shufflesRegister,
      Map.Entry<SecureProfileUid, SecureProfileResolveRequest> entry) {
    final SecureProfileUid queriedBy = entry.getKey();
    final SecureProfileFutureUID futureUID = shufflesRegister.get(queriedBy);
    if (futureUID == null) {
      cache.populate(queriedBy.toString(), FutureShufflesCache.PHANTOM_ENTRY);
      return;
    }

    final SecureProfileUid resolvedId = futureUID.getResolved();
    final SecureProfileResolveRequest request = entry.getValue();

    final String uniqueId = futureUID.getUniqueId();
    final ResolvedId resolvedProfileUID = ResolvedId.create(resolvedId.toString(), uniqueId, DEVICE_PROFILE);

    // after remove old entries from cache (@evictOutdatedCacheEntries), add the new ones
    if (ResolverType.CACHE != request.getResolvedBy()) {
      addNewCacheEntries(futureUID, resolvedProfileUID);
    }

    if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
      notifyListeners(request, resolvedProfileUID);
    }
  }

  private void addNewCacheEntries(final SecureProfileFutureUID futureUID, final ResolvedId resolvedId) {
    for (SecureProfileUid futureShuffle : futureUID.getFutureIds()) {
      cache.populate(futureShuffle.toString(), resolvedId);
    }
    cache.populate(resolvedId.getSecureProfileUID().toString(), resolvedId);
  }

  private void notifyListeners(final SecureProfileResolveRequest request, final ResolvedId resolvedId) {
    final ISecureProfile fakeDevice = request.getFakeDevice();
    final ISecureProfile resolvedDevice = new SecureProfile.Builder(fakeDevice).resolvedId(resolvedId).build();
    cache.notifyListeners(resolvedDevice);
  }
}
