package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

/**
 * An abstract representation of a region. Venue acts as a physical location and allows us to group
 * {@link Device} objects together. Additionally it's a describable container for devices which can
 * be signified with image cover. A venue belongs to the {@link Manager} that created it and can be
 * shared to the other ones by his owner.
 * <br>
 * We can obtain venues via {@link KontaktCloud}.
 * <br>
 * To create new instance of this class, please use {@link Venue.Builder}.
 */
public class Venue implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Venue> CREATOR = new Creator<Venue>() {
    @Override
    public Venue createFromParcel(Parcel source) {
      return new Venue(source);
    }

    @Override
    public Venue[] newArray(int size) {
      return new Venue[size];
    }
  };

  private UUID id;
  private String name;
  private String description;
  private int devicesCount;
  private List<Device> devices;
  private String coverType;
  private String image;
  private UUID managerId;
  private Access access;
  private String lng;
  private String lat;

  public static Builder builder() {
    return new Builder();
  }

  private Venue() {
    this(new Builder());
  }

  Venue(Builder builder) {
    id = builder.id;
    name = builder.name;
    description = builder.description;
    devicesCount = builder.devicesCount;
    devices = builder.devices;
    coverType = builder.coverType;
    image = builder.imageUrl;
    managerId = builder.managerId;
    lng = builder.lng;
    lat = builder.lat;
    access = builder.access;
  }

  protected Venue(Parcel in) {
    this.id = (UUID) in.readSerializable();
    this.name = in.readString();
    this.description = in.readString();
    this.devicesCount = in.readInt();
    this.devices = in.createTypedArrayList(Device.CREATOR);
    this.coverType = in.readString();
    this.image = in.readString();
    this.managerId = (UUID) in.readSerializable();
    int tmpAccess = in.readInt();
    this.access = tmpAccess == -1 ? null : Access.values()[tmpAccess];
    this.lng = in.readString();
    this.lat = in.readString();
  }

  @Override
  public boolean equals(Object o) {
    if (o == this) {
      return true;
    }

    if (o == null || !(o instanceof Venue)) {
      return false;
    }

    final Venue venue = (Venue) o;

    return SDKEqualsBuilder.start()
        .equals(id, venue.id)
        .equals(name, venue.name)
        .equals(description, venue.description)
        .equals(coverType, venue.coverType)
        .equals(devicesCount, venue.devicesCount)
        .equals(image, venue.image)
        .equals(devices, venue.devices)
        .equals(managerId, venue.managerId)
        .equals(lat, venue.lat)
        .equals(lng, venue.lng)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(id)
        .append(name)
        .append(description)
        .append(coverType)
        .append(devicesCount)
        .append(image)
        .append(devices)
        .append(managerId)
        .append(lat)
        .append(lng)
        .build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.id);
    dest.writeString(this.name);
    dest.writeString(this.description);
    dest.writeInt(this.devicesCount);
    dest.writeTypedList(devices);
    dest.writeString(this.coverType);
    dest.writeString(this.image);
    dest.writeSerializable(this.managerId);
    dest.writeInt(this.access == null ? -1 : this.access.ordinal());
    dest.writeString(this.lng);
    dest.writeString(this.lat);
  }

  public UUID getId() {
    return id;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public int getDevicesCount() {
    return devicesCount;
  }

  public String getCoverType() {
    return coverType;
  }

  public String getImageUrl() {
    return image;
  }

  public UUID getManagerId() {
    return managerId;
  }

  public List<Device> getDevices() {
    return devices;
  }

  public Access getAccess() {
    return access;
  }

  public String getLongitude() {
    return lng;
  }

  public String getLatitude() {
    return lat;
  }

  public static class Builder {
    UUID id;
    String name;
    String description;
    int devicesCount;
    List<Device> devices = new ArrayList<>();
    String coverType;
    String imageUrl;
    UUID managerId;
    String lng;
    String lat;
    Access access;

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder description(final String description) {
      this.description = description;
      return this;
    }

    public Builder devicesCount(final int devicesCount) {
      SDKPreconditions.checkState(devicesCount >= 0, "devices count cannot be negative");
      this.devicesCount = devicesCount;
      return this;
    }

    public Builder addDevices(final Collection<Device> devices) {
      SDKPreconditions.checkNotNull(devices, "devices cannot be null");
      for (Device device : devices) {
        SDKPreconditions.checkNotNull(device, "devices cannot contain null value");
      }
      this.devices.addAll(devices);
      return this;
    }

    public Builder addDevice(final Device device) {
      SDKPreconditions.checkNotNull(device, "device cannot be null");
      this.devices.add(device);
      return this;
    }

    public Builder coverType(final String coverType) {
      this.coverType = coverType;
      return this;
    }

    public Builder imageUrl(final String imageUrl) {
      this.imageUrl = imageUrl;
      return this;
    }

    public Builder managerId(final UUID managerId) {
      this.managerId = managerId;
      return this;
    }

    public Builder longitude(final String lng) {
      this.lng = lng;
      return this;
    }

    public Builder latitude(final String lat) {
      this.lat = lat;
      return this;
    }

    public Builder access(final Access access) {
      this.access = access;
      return this;
    }

    public Venue build() {
      return new Venue(this);
    }
  }
}
