package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Model class that represents JSON structure of Kontakt.io place.
 */
public class Place implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Parcelable.Creator<Place> CREATOR = new Parcelable.Creator<Place>() {
    @Override
    public Place createFromParcel(Parcel source) {
      return new Place(source);
    }

    @Override
    public Place[] newArray(int size) {
      return new Place[size];
    }
  };

  private final UUID id;
  private final String name;
  private final UUID managerId;
  private final List<String> receiverIds;
  private final UUID venueId;
  private final UUID parentId;
  private final String schema;
  private final String schemaType;
  private final Coordinates coordinates;
  private final Coordinates geoCoordinates;
  private final double scale;

  public static Builder builder() {
    return new Builder();
  }

  private Place() {
    this(new Builder());
  }

  Place(Builder builder) {
    this.id = builder.id;
    this.name = builder.name;
    this.managerId = builder.managerId;
    this.receiverIds = builder.receiverIds;
    this.venueId = builder.venueId;
    this.parentId = builder.parentId;
    this.schema = builder.schema;
    this.schemaType = builder.schemaType;
    this.coordinates = builder.coordinates;
    this.geoCoordinates = builder.geoCoordinates;
    this.scale = builder.scale;
  }

  protected Place(Parcel in) {
    this.id = (UUID) in.readSerializable();
    this.name = in.readString();
    this.managerId = (UUID) in.readSerializable();
    this.receiverIds = in.createStringArrayList();
    this.venueId = (UUID) in.readSerializable();
    this.parentId = (UUID) in.readSerializable();
    this.schema = in.readString();
    this.schemaType = in.readString();
    this.coordinates = in.readParcelable(Coordinates.class.getClassLoader());
    this.geoCoordinates = in.readParcelable(Coordinates.class.getClassLoader());
    this.scale = in.readDouble();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    Place place = (Place) o;

    if (Double.compare(place.scale, scale) != 0) return false;
    if (id != null ? !id.equals(place.id) : place.id != null) return false;
    if (name != null ? !name.equals(place.name) : place.name != null) return false;
    if (managerId != null ? !managerId.equals(place.managerId) : place.managerId != null) return false;
    if (receiverIds != null ? !receiverIds.equals(place.receiverIds) : place.receiverIds != null) return false;
    if (venueId != null ? !venueId.equals(place.venueId) : place.venueId != null) return false;
    if (parentId != null ? !parentId.equals(place.parentId) : place.parentId != null) return false;
    if (schema != null ? !schema.equals(place.schema) : place.schema != null) return false;
    if (schemaType != null ? !schemaType.equals(place.schemaType) : place.schemaType != null) return false;
    if (coordinates != null ? !coordinates.equals(place.coordinates) : place.coordinates != null) return false;
    return geoCoordinates != null ? geoCoordinates.equals(place.geoCoordinates) : place.geoCoordinates == null;
  }

  @Override
  public int hashCode() {
    int result;
    long temp;
    result = id != null ? id.hashCode() : 0;
    result = 31 * result + (name != null ? name.hashCode() : 0);
    result = 31 * result + (managerId != null ? managerId.hashCode() : 0);
    result = 31 * result + (receiverIds != null ? receiverIds.hashCode() : 0);
    result = 31 * result + (venueId != null ? venueId.hashCode() : 0);
    result = 31 * result + (parentId != null ? parentId.hashCode() : 0);
    result = 31 * result + (schema != null ? schema.hashCode() : 0);
    result = 31 * result + (schemaType != null ? schemaType.hashCode() : 0);
    result = 31 * result + (coordinates != null ? coordinates.hashCode() : 0);
    result = 31 * result + (geoCoordinates != null ? geoCoordinates.hashCode() : 0);
    temp = Double.doubleToLongBits(scale);
    result = 31 * result + (int) (temp ^ (temp >>> 32));
    return result;
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.id);
    dest.writeString(this.name);
    dest.writeSerializable(this.managerId);
    dest.writeStringList(this.receiverIds);
    dest.writeSerializable(this.venueId);
    dest.writeSerializable(this.parentId);
    dest.writeString(this.schema);
    dest.writeString(this.schemaType);
    dest.writeParcelable(this.coordinates, flags);
    dest.writeParcelable(this.geoCoordinates, flags);
    dest.writeDouble(this.scale);
  }

  public UUID getId() {
    return id;
  }

  public String getName() {
    return name;
  }

  public UUID getManagerId() {
    return managerId;
  }

  public List<String> getReceiverIds() {
    return receiverIds;
  }

  public UUID getVenueId() {
    return venueId;
  }

  public UUID getParentId() {
    return parentId;
  }

  public String getSchema() {
    return schema;
  }

  public String getSchemaType() {
    return schemaType;
  }

  public Coordinates getCoordinates() {
    return coordinates;
  }

  public GeoCoordinates getGeoCoordinates() {
    return (GeoCoordinates) geoCoordinates;
  }

  public double getScale() {
    return scale;
  }

  public static class Builder {
    UUID id;
    String name;
    UUID managerId;
    List<String> receiverIds = new ArrayList<>();
    UUID venueId;
    UUID parentId;
    String schema;
    String schemaType;
    Coordinates coordinates;
    Coordinates geoCoordinates;
    double scale;

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder managerId(final UUID managerId) {
      this.managerId = managerId;
      return this;
    }

    public Builder addReceiverIds(final Collection<String> receiverIds) {
      checkNotNull(receiverIds, "receiver IDs cannot be null");
      this.receiverIds.addAll(receiverIds);
      return this;
    }

    public Builder venueId(final UUID venueId) {
      this.venueId = venueId;
      return this;
    }

    public Builder parentId(final UUID parentId) {
      this.parentId = parentId;
      return this;
    }

    public Builder schema(final String schema) {
      this.schema = schema;
      return this;
    }

    public Builder schemaType(final String schemaType) {
      this.schemaType = schemaType;
      return this;
    }

    public Builder coordinates(final Coordinates coordinates) {
      this.coordinates = coordinates;
      return this;
    }

    public Builder geoCoordinates(final GeoCoordinates geoCoordinates) {
      this.geoCoordinates = geoCoordinates;
      return this;
    }

    public Builder scale(final double scale) {
      this.scale = scale;
      return this;
    }

    public Place build() {
      return new Place(this);
    }
  }
}
