package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;

import java.util.Arrays;
import java.util.EnumSet;

import static com.kontakt.sdk.android.common.model.Network.Type.NONE;
import static com.kontakt.sdk.android.common.model.Network.Type.WEP;
import static com.kontakt.sdk.android.common.model.Network.Type.WPA2_ENTERPRISE;
import static com.kontakt.sdk.android.common.model.Network.Type.WPA2_PERSONAL;
import static com.kontakt.sdk.android.common.model.Network.Type.WPA_ENTERPRISE;
import static com.kontakt.sdk.android.common.model.Network.Type.WPA_PERSONAL;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static java.util.Arrays.copyOfRange;

/**
 * Model class representing Gateway's scanned WiFi network.
 */
public class Network implements Parcelable {

  private static final int BLE_NETWORK_MIN_LENGTH = 6;
  private static final int SSID_INDEX = 6;
  private static final int ENTERPRISE_MASK = 0b0000_0010;

  private final int signalStrength;
  private final boolean isSecure;
  private final String name;
  private final String login;
  private final String password;
  private final String apiKey;
  private final EnumSet<Type> type;

  public static Network fromBleValue(byte[] value) {
    checkArgument(value.length > BLE_NETWORK_MIN_LENGTH, "Unexpected frame length. Should be at least: " + BLE_NETWORK_MIN_LENGTH);

    byte wpaFlag = value[3];
    byte rsnFlag = value[5];

    boolean isSecure = (value[1] == 1);

    EnumSet<Type> type = EnumSet.noneOf(Type.class);
    if (isSecure) {
      if (wpaFlag == 0 && rsnFlag == 0) {
        type.add(WEP);
      }

      if ((wpaFlag & ENTERPRISE_MASK) != 0) {
        type.add(WPA_ENTERPRISE);
      } else if (wpaFlag != 0) {
        type.add(WPA_PERSONAL);
      }

      if ((rsnFlag & ENTERPRISE_MASK) != 0) {
        type.add(WPA2_ENTERPRISE);
      } else if (rsnFlag != 0) {
        type.add(WPA2_PERSONAL);
      }
    } else {
      type.add(NONE);
    }

    return new Builder()
        .signalStrength(value[0])
        .isSecure(isSecure)
        .type(type)
        .name(new String(copyOfRange(value, SSID_INDEX, value.length)))
        .build();
  }

  public Network(Builder builder) {
    this.signalStrength = builder.signalStrength;
    this.isSecure = builder.isSecure;
    this.name = builder.name;
    this.login = builder.login;
    this.password = builder.password;
    this.type = builder.type;
    this.apiKey = builder.apiKey;
  }

  private Network() {
    //For GSON
    this(new Builder());
  }

  public int getSignalStrength() {
    return signalStrength;
  }

  public boolean isSecure() {
    return isSecure;
  }

  public boolean isEnterprise() {
    return type != null && (type.contains(WPA2_ENTERPRISE) || type.contains(WPA_ENTERPRISE));
  }

  public boolean isPersonal() {
    return type != null && (type.contains(WPA2_PERSONAL) || type.contains(WPA_PERSONAL));
  }

  public String getName() {
    return name;
  }

  public String getLogin() {
    return login;
  }

  public String getPassword() {
    return password;
  }

  public String getApiKey() {
    return apiKey;
  }

  public EnumSet<Type> getType() {
    return type;
  }

  public static class Builder {

    int signalStrength;
    boolean isSecure;
    String name;
    String password;
    String apiKey;
    String login;
    EnumSet<Type> type;

    public Builder signalStrength(int signalStrength) {
      this.signalStrength = signalStrength;
      return this;
    }

    public Builder isSecure(boolean secure) {
      this.isSecure = secure;
      return this;
    }

    public Builder name(String name) {
      this.name = name;
      return this;
    }

    public Builder login(String login) {
      this.login = login;
      return this;
    }

    public Builder password(String password) {
      this.password = password;
      return this;
    }

    public Builder apiKey(String apiKey) {
      this.apiKey = apiKey;
      return this;
    }

    public Builder type(EnumSet<Type> type) {
      this.type = type;
      return this;
    }

    public Network build() {
      return new Network(this);
    }
  }

  public enum Type {
    WEP,
    WPA_PERSONAL,
    WPA_ENTERPRISE,
    WPA2_PERSONAL,
    WPA2_ENTERPRISE,
    NONE
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    Network network = (Network) o;

    if (signalStrength != network.signalStrength) return false;
    if (isSecure != network.isSecure) return false;
    if (name != null ? !name.equals(network.name) : network.name != null) return false;
    if (login != null ? !login.equals(network.login) : network.login != null) return false;
    if (password != null ? !password.equals(network.password) : network.password != null) return false;
    if (apiKey != null ? !apiKey.equals(network.apiKey) : network.apiKey != null) return false;
    return type != null ? type.equals(network.type) : network.type == null;

  }

  @Override
  public int hashCode() {
    int result = signalStrength;
    result = 31 * result + (isSecure ? 1 : 0);
    result = 31 * result + (name != null ? name.hashCode() : 0);
    result = 31 * result + (login != null ? login.hashCode() : 0);
    result = 31 * result + (password != null ? password.hashCode() : 0);
    result = 31 * result + (apiKey != null ? apiKey.hashCode() : 0);
    result = 31 * result + (type != null ? type.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "Network{" +
        "signalStrength=" + signalStrength +
        ", isSecure=" + isSecure +
        ", name='" + name + '\'' +
        ", login='" + login + '\'' +
        ", password='" + password + '\'' +
        ", apiKey='" + apiKey + '\'' +
        ", type=" + Arrays.toString(type.toArray()) +
        '}';
  }

  //Parcelable

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.signalStrength);
    dest.writeByte(this.isSecure ? (byte) 1 : (byte) 0);
    dest.writeString(this.name);
    dest.writeString(this.login);
    dest.writeString(this.password);
    dest.writeString(this.apiKey);
    dest.writeSerializable(this.type);
  }

  protected Network(Parcel in) {
    this.signalStrength = in.readInt();
    this.isSecure = in.readByte() != 0;
    this.name = in.readString();
    this.login = in.readString();
    this.password = in.readString();
    this.apiKey = in.readString();
    this.type = (EnumSet<Type>) in.readSerializable();
  }

  public static final Creator<Network> CREATOR = new Creator<Network>() {
    @Override
    public Network createFromParcel(Parcel source) {
      return new Network(source);
    }

    @Override
    public Network[] newArray(int size) {
      return new Network[size];
    }
  };
}
