package com.kontakt.sdk.android.cloud.api.executor.configs;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ConfigsApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.Network;
import com.kontakt.sdk.android.common.model.PacketType;
import com.kontakt.sdk.android.common.model.PowerSaving;
import com.kontakt.sdk.android.common.model.PowerSavingFeature;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import retrofit2.Call;

/**
 * Request executor provided by {@link ConfigsApi}. Use this class if you want to create
 * configs through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Config config = new Config.Builder().major(532).minor(111).build();
 *   kontaktCloud.configs().create(config)
 *      .forDevices("AxT7", "GVy6", "9KiJ")
 *      .withType(DeviceType.BEACON)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that devices and device type must be specified so invocations of {@code forDevices}
 * and {@code withType} methods are mandatory. Otherwise an exception will be thrown.
 */
public class CreateConfigRequestExecutor extends RequestExecutor<Config[]> {

  private final ConfigsService configsService;

  private final Config config;
  private String[] uniqueIds;
  private DeviceType type;

  /**
   * Constructs request executor initialized with corresponding service class and new config object.
   *
   * @param configsService the configs API service.
   * @param config         new config.
   */
  public CreateConfigRequestExecutor(final ConfigsService configsService, final Config config) {
    this.configsService = configsService;
    this.config = config;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CreateConfigRequestExecutor forDevices(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CreateConfigRequestExecutor forDevices(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies device type. The method invocation is obligatory while using this request executor.
   *
   * @param type the device type.
   * @return this request executor.
   */
  public CreateConfigRequestExecutor withType(final DeviceType type) {
    this.type = SDKPreconditions.checkNotNull(type, "type cannot be null");
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Config[] execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Config[]> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Config[]> prepareCall() {
    return configsService.createConfig(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(uniqueIds != null, "specify devices");
    SDKPreconditions.checkState(type != null, "specify device type");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    params.put(CloudConstants.Configs.DEVICE_TYPE_PARAMETER, type.name());
    params.put(CloudConstants.Configs.SHUFFLED_PARAMETER, String.valueOf(config.isShuffled()));

    if (config.getName() != null) {
      params.put(CloudConstants.Configs.NAME_PARAMETER, config.getName());
    }
    if (config.getProximity() != null) {
      params.put(CloudConstants.Configs.PROXIMITY_PARAMETER, config.getProximity().toString());
    }
    if (config.getMajor() != -1) {
      params.put(CloudConstants.Configs.MAJOR_PARAMETER, String.valueOf(config.getMajor()));
    }
    if (config.getMinor() != -1) {
      params.put(CloudConstants.Configs.MINOR_PARAMETER, String.valueOf(config.getMinor()));
    }
    if (config.getNamespace() != null) {
      params.put(CloudConstants.Configs.NAMESPACE_PARAMETER, config.getNamespace());
    }
    if (config.getInstanceId() != null) {
      params.put(CloudConstants.Configs.INSTANCE_ID_PARAMETER, config.getInstanceId());
    }
    if (config.getUrl() != null) {
      params.put(CloudConstants.Configs.URL_PARAMETER, config.getHexUrl());
    }
    if (config.getTxPower() != -1) {
      params.put(CloudConstants.Configs.TX_POWER_PARAMETER, String.valueOf(config.getTxPower()));
    }
    if (config.getInterval() != -1) {
      params.put(CloudConstants.Configs.INTERVAL_PARAMETER, String.valueOf(config.getInterval()));
    }
    if (config.getPassword() != null) {
      params.put(CloudConstants.Configs.PASSWORD_PARAMETER, config.getPassword());
    }
    if (config.getPowerSaving() != null) {
      addPowerSavingParams(params, config.getPowerSaving());
    }
    if (config.getGatewayNetwork() != null) {
      addGatewayNetworkParams(params, config.getGatewayNetwork());
    }

    List<DeviceProfile> profiles = config.getProfiles();
    if (profiles != null && !profiles.isEmpty()) {
      params.put(CloudConstants.Configs.PROFILES_PARAMETER, StringUtils.join(profiles, ","));
    }

    List<PacketType> packets = config.getPackets();
    if (packets != null && !packets.isEmpty()) {
      params.put(CloudConstants.Configs.PACKETS_PARAMETER, StringUtils.join(packets, ","));
    }

    List<Integer> rssi1m = config.getRssi1m();
    if (rssi1m != null && !rssi1m.isEmpty()) {
      params.put(CloudConstants.Configs.RSSI_1M_PARAMETER, StringUtils.join(rssi1m, ","));
    }

    List<Integer> rssi0m = config.getRssi0m();
    if (rssi0m != null && !rssi0m.isEmpty()) {
      params.put(CloudConstants.Configs.RSSI_0M_PARAMETER, StringUtils.join(rssi0m, ","));
    }

    if (config.getTemperatureOffset() != Config.TEMPERATURE_OFFSET_FEATURE_NOT_SUPPORTED_VALUE) {
      params.put(CloudConstants.Configs.TEMPERATURE_OFFSET_PARAMETER, String.valueOf(config.getTemperatureOffset()));
    }

    return params;
  }

  private void addGatewayNetworkParams(Map<String, String> params, Network network) {
    if (network.getName() != null) {
      params.put(CloudConstants.Configs.GATEWAY_NETWORK_SSID, network.getName());
    }
    if (network.getLogin() != null) {
      params.put(CloudConstants.Configs.GATEWAY_NETWORK_LOGIN, network.getLogin());
    }
    if (network.getPassword() != null) {
      params.put(CloudConstants.Configs.GATEWAY_NETWORK_PASSWORD, network.getPassword());
    }
    if (network.getApiKey() != null) {
      params.put(CloudConstants.Configs.GATEWAY_NETWORK_APIKEY, network.getApiKey());
    }
    if (network.getType() != null && !network.getType().contains(Network.Type.NONE)) {
      if (network.isEnterprise()) {
        params.put(CloudConstants.Configs.GATEWAY_NETWORK_PROTOCOL, "WPA_EAP");
      } else if (network.isPersonal()) {
        params.put(CloudConstants.Configs.GATEWAY_NETWORK_PROTOCOL, "WPA_PSK");
      }
    }
  }

  private void addPowerSavingParams(Map<String, String> params, PowerSaving powerSaving) {
    int lightSensorHysteresis = powerSaving.getLightSensorHysteresis();
    if (lightSensorHysteresis != -1) {
      params.put(asPowerSavingParam(CloudConstants.Configs.LIGHT_SENSOR_HYSTERESIS_PARAMETER), String.valueOf(lightSensorHysteresis));
    }

    long lightSensorSamplingInterval = powerSaving.getLightSensorSamplingInterval();
    if (lightSensorSamplingInterval != -1) {
      params.put(asPowerSavingParam(CloudConstants.Configs.LIGHT_SENSOR_SAMPLING_INTERVAL_PARAMETER), String.valueOf(lightSensorSamplingInterval));
    }

    int lightSensorThreshold = powerSaving.getLightSensorThreshold();
    if (lightSensorThreshold != -1) {
      params.put(asPowerSavingParam(CloudConstants.Configs.LIGHT_SENSOR_THRESHOLD_PARAMETER), String.valueOf(lightSensorThreshold));
    }

    long moveSuspendTimeout = powerSaving.getMoveSuspendTimeout();
    if (moveSuspendTimeout > 0) {
      params.put(asPowerSavingParam(CloudConstants.Configs.MOVE_SUSPEND_TIMEOUT_PARAMETER), String.valueOf(moveSuspendTimeout));
    }

    List<PowerSavingFeature> features = powerSaving.getFeatures();
    if (features != null) {
      params.put(asPowerSavingParam(CloudConstants.Configs.FEATURES_PARAMETER), StringUtils.join(features, ","));
    }
  }

  private String asPowerSavingParam(String param) {
    return StringUtils.join(Arrays.asList(CloudConstants.Configs.POWER_SAVING_PREFIX, param), ".");
  }
}
