package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.os.Build;

import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.concurrent.TimeUnit;

@SuppressWarnings("MissingPermission")
final class Runners {

  private Runners() {
    // prevent from creation
  }

  static Runnable newRunner(final RunnerType runnerType, ScanConfiguration configuration) {
    SDKPreconditions.checkNotNull(runnerType, "Runner type cannot be null");
    SDKPreconditions.checkNotNull(configuration, "Configuration cannot be null");
    switch (runnerType) {
      case MONITOR_ACTIVE_RUNNER:
        return newMonitorActiveRunner(configuration);
      case MONITOR_PASSIVE_RUNNER:
        return newMonitorPassiveRunner(configuration);
      case FORCE_SCAN_RUNNER:
        return newForceScanRunner(configuration);
      default:
        throw new RuntimeException();
    }
  }

  private static Runnable newForceScanRunner(final ScanConfiguration configuration) {
    final MonitorCallback monitorCallback = (MonitorCallback) configuration.getScanCallback();
    final ForceScanConfiguration forceScanConfiguration = configuration.getScanContext().getForceScanConfiguration();
    final long forceScanActivePeriod = forceScanConfiguration.getForceScanActivePeriod();
    final long forceScanPassivePeriod = forceScanConfiguration.getForceScanPassivePeriod();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
      @Override
      public void run() {
        final BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        try {
          if (bluetoothAdapter == null) {
            onBluetoothAdapterNotInitialized(monitorCallback);
            return;
          }

          while (!Thread.currentThread().isInterrupted()) {
            Logger.d("Stopping Scan (force)");
            bluetoothAdapter.stopLeScan(monitorCallback);

            Logger.d(String.format("Sleep during passive period: %s", String.valueOf(forceScanPassivePeriod)));
            TimeUnit.MILLISECONDS.sleep(forceScanPassivePeriod);

            Logger.d("Starting scan (force)");
            bluetoothAdapter.startLeScan(monitorCallback);

            Logger.d(String.format("Sleep during active period: %s", String.valueOf(forceScanActivePeriod)));
            TimeUnit.MILLISECONDS.sleep(forceScanActivePeriod);
          }

          bluetoothAdapter.stopLeScan(monitorCallback);
          Logger.d("Force scan finished");
        } catch (InterruptedException e) {
          bluetoothAdapter.stopLeScan(monitorCallback);
          Logger.d("Force scan interrupted");
        }
      }
    };
  }

  private static Runnable newMonitorPassiveRunner(final ScanConfiguration configuration) {
    final MonitorCallback monitorCallback = (MonitorCallback) configuration.getScanCallback();

    return new Runnable() {

      @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
      @Override
      public void run() {
        BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        if (bluetoothAdapter == null) {
          onBluetoothAdapterNotInitialized(monitorCallback);
          return;
        }

        bluetoothAdapter.stopLeScan(monitorCallback);
        monitorCallback.onMonitorCycleStop();
      }
    };
  }

  private static Runnable newMonitorActiveRunner(final ScanConfiguration serviceConfiguration) {
    final MonitorCallback monitorCallback = (MonitorCallback) serviceConfiguration.getScanCallback();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
      @Override
      public void run() {
        final BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        if (bluetoothAdapter == null) {
          onBluetoothAdapterNotInitialized(monitorCallback);
          return;
        }

        bluetoothAdapter.startLeScan(monitorCallback);
        monitorCallback.onMonitorCycleStart();
      }
    };
  }

  private static void onBluetoothAdapterNotInitialized(final MonitorCallback monitorCallback) {
    final String message = "Unexpected error occurred - BluetoothAdapter is null.";
    Logger.e(message);
    monitorCallback.onScanError(new ScanError(message));
  }

  enum RunnerType {
    /**
     * The MONITOR_ACTIVE_RUNNER.
     */
    MONITOR_ACTIVE_RUNNER,

    /**
     * The MONITOR_PASSIVE_RUNNER.
     */
    MONITOR_PASSIVE_RUNNER,

    /**
     * The FORCE_SCAN_RUNNER.
     */
    FORCE_SCAN_RUNNER
  }

}
