package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.os.Build;
import android.util.SparseLongArray;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.ShuffledDevicesResolver;
import com.kontakt.sdk.android.ble.discovery.ShuffledSecureProfileResolver;
import com.kontakt.sdk.android.ble.util.SafeSparseLongArray;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewDiscovered;
import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewLost;
import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewUpdated;
import static com.kontakt.sdk.android.common.util.SecureProfileUtils.asRemoteBluetoothDevice;
import static com.kontakt.sdk.android.common.util.SecureProfileUtils.fromRemoteBluetoothDevice;
import static java.lang.System.currentTimeMillis;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class SecureProfileDiscoverer implements BluetoothDeviceDiscoverer, ShuffledSecureProfileResolver.ResolveCallback {

  //private final ShuffledDevicesResolver shuffleResolver;
  private final ShuffledSecureProfileResolver shuffleResolver;

  private final Map<String, ISecureProfile> cache; //Mac Address -> ISecureProfile
  private final SecureProfileParser parser;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final SparseLongArray secureProfileTimestampArray;
  private final DiscoveryContract discoveryContract;
  private final long deviceUpdateIntervalCallback;
  private long lastUpdateCallbackMillis;

  public SecureProfileDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract, FutureShufflesCache shufflesCache) {
    this.deviceUpdateIntervalCallback = scanContext.getDeviceUpdateCallbackInterval();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.secureProfileTimestampArray = new SafeSparseLongArray();
    this.discoveryContract = discoveryContract;
    this.parser = new SecureProfileParser(scanContext);
    this.cache = new ConcurrentHashMap<>();

    //this.shuffleResolver = new ShuffledDevicesResolver(this, shufflesCache);
    this.shuffleResolver = new ShuffledSecureProfileResolver(this, shufflesCache);
  }

  // used for tests
  SecureProfileDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract, SparseLongArray secureProfileTimestampArray,
                          Map<String, ISecureProfile> cache, FutureShufflesCache shufflesCache) {
    this.discoveryContract = discoveryContract;
    this.deviceUpdateIntervalCallback = scanContext.getDeviceUpdateCallbackInterval();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.secureProfileTimestampArray = secureProfileTimestampArray;
    this.parser = new SecureProfileParser(scanContext);
    this.cache = cache;

    //this.shuffleResolver = new ShuffledDevicesResolver(this, shufflesCache);
    this.shuffleResolver = new ShuffledSecureProfileResolver(this, shufflesCache);
  }

  @Override
  public void performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanResult) {
    if (!parser.isValidSecureProfileFrame(scanResult)) {
      return;
    }

    parser.parseScanRecord(scanResult);
    ISecureProfile secureProfile = parser.getSecureProfile(bluetoothDevice, rssi);
    if (secureProfile == null) {
      Logger.w("Unsupported SecureProfile frame type. Skipping.");
      return;
    }

    notifySecureProfilePresent(secureProfile);
    if (secureProfile.isShuffled()) {
      shuffleResolver.resolve(secureProfile);
    } else {
      handleResolved(secureProfile);
    }
  }

  @Override
  public void onResolved(ISecureProfile secureProfile) {
    handleResolved(secureProfile);
  }

  private void handleResolved(ISecureProfile secureProfile) {
    boolean isDiscovered = cache.containsKey(secureProfile.getMacAddress());
    cache.put(secureProfile.getMacAddress(), secureProfile);

    if (!isDiscovered) {
      onDiscovered(secureProfile);
    } else {
      onUpdated();
    }
  }

  @Override
  public void evictInactiveDevices(long currentTimeMillis) {
    final long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();

    for (ISecureProfile secureProfile : new ArrayList<>(cache.values())) {
      long timestamp = getSecureProfileTimestamp(secureProfile);
      if (timestamp != -1L) {
        long threshold = currentTimeMillis - timestamp;
        if (threshold > inactivityTimeout) {
          cache.remove(secureProfile.getMacAddress());
          clearBeforeLost(secureProfile);
          onLost(secureProfile);
          removeSecureProfileTimestamp(secureProfile);
        }
      }
    }
  }

  private void notifySecureProfilePresent(ISecureProfile secureProfile) {
    secureProfileTimestampArray.put(secureProfile.getMacAddress().hashCode(), currentTimeMillis());
  }

  private void onDiscovered(ISecureProfile secureProfile) {
    discoveryContract.onEvent(createNewDiscovered(secureProfile, currentTimeMillis()));
  }

  private void onUpdated() {
    long currentTimeMillis = currentTimeMillis();
    if (currentTimeMillis - lastUpdateCallbackMillis > deviceUpdateIntervalCallback) {
      discoveryContract.onEvent(createNewUpdated(new ArrayList<>(cache.values()), currentTimeMillis));
      lastUpdateCallbackMillis = currentTimeMillis;
    }
  }

  private void onLost(ISecureProfile secureProfile) {
    discoveryContract.onEvent(createNewLost(secureProfile, currentTimeMillis()));

    //shuffleResolver.onDeviceLost(asRemoteBluetoothDevice(secureProfile));
    shuffleResolver.onDeviceLost(secureProfile);
  }

  private void clearBeforeLost(ISecureProfile secureProfile) {
    parser.clearRssiCalculation(secureProfile.getMacAddress().hashCode());
  }

  private long getSecureProfileTimestamp(ISecureProfile secureProfile) {
    return secureProfileTimestampArray.get(secureProfile.getMacAddress().hashCode(), -1L);
  }

  private void removeSecureProfileTimestamp(ISecureProfile secureProfile) {
    int index = secureProfileTimestampArray.indexOfKey(secureProfile.getMacAddress().hashCode());
    if (index >= 0) {
      secureProfileTimestampArray.removeAt(index);
    }
  }

  @Override
  public void disable() {
    parser.disable();
    shuffleResolver.disable();
  }
}
