package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.ble.device.EddystoneDevice;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.EddystoneFutureUID;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;

class EddystoneUIDResolver implements Runnable {

  private static final String TAG = "EddystoneUIDResolver";
  private static final int REQUEST_UNIT_SIZE = 120;
  private static final int DEFAULT_BUFFER_SIZE = 200;
  private static final DeviceProfile DEVICE_PROFILE = DeviceProfile.EDDYSTONE;
  private final ArrayBlockingQueue<EddystoneResolveRequest> requestQueue;
  private final FutureShufflesCache cache;
  private final Collection<EddystoneResolveStrategy> strategies;

  EddystoneUIDResolver(FutureShufflesCache futureShufflesCache, KontaktCloud kontaktCloud) {
    this.cache = futureShufflesCache;
    this.requestQueue = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
    List<EddystoneResolveStrategy> resolvers = new LinkedList<>();
    EddystoneResolveStrategy cacheStrategy = new EddystoneCacheResolveStrategy(cache);
    EddystoneResolveStrategy apiStrategy = new EddystoneApiResolveStrategy(kontaktCloud);
    resolvers.add(cacheStrategy);
    resolvers.add(apiStrategy);
    this.strategies = Collections.unmodifiableCollection(resolvers);
  }

  @Override
  public void run() {
    if (!cache.isInitialized()) {
      Logger.d(TAG + " Cache not initialized yet");
      return;
    }
    List<EddystoneResolveRequest> requests = new ArrayList<>();
    requestQueue.drainTo(requests, REQUEST_UNIT_SIZE);
    if (requests.isEmpty()) {
      Logger.d(TAG + " Nothing to resolve");
      return;
    }
    try {
      Logger.d(TAG + " Start resolving");

      final Map<EddystoneUid, EddystoneResolveRequest> requestsRegister = buildRequestsRegister(requests);
      final List<EddystoneFutureUID> shuffles = resolveShuffles(requestsRegister);
      final Map<EddystoneUid, EddystoneFutureUID> shufflesRegister = buildShufflesRegister(shuffles);

      evictOutdatedCacheEntries(requestsRegister, shufflesRegister);
      handleRequests(requestsRegister, shufflesRegister);

      cache.serialize();
    } catch (Exception e) {
      Throwable cause = e.getCause();
      if (UnknownHostException.class.isInstance(cause) || SocketTimeoutException.class.isInstance(cause)) {
        requestQueue.addAll(requests);
      } else {
        Logger.e(TAG + " Error occurs when try to resolve shuffled device ", e);
      }
    }
  }

  void addResolveRequest(EddystoneResolveRequest request) {
    if (requestQueue.contains(request)) {
      return;
    }
    try {
      requestQueue.add(request);
    } catch (IllegalStateException fullQueueException) {
      Logger.e("Could not add Eddystone to resolve", fullQueueException);
    }
  }

  public void markIgnored(RemoteBluetoothDevice beacon) {
    for (EddystoneResolveRequest request : requestQueue) {
      if (request.getFakeDevice().equals(beacon)) {
        request.setStatus(ResolveRequestStatus.IGNORED);
      }
    }
  }

  public void clear() {
    requestQueue.clear();
  }

  private Map<EddystoneUid, EddystoneResolveRequest> buildRequestsRegister(final List<EddystoneResolveRequest> requests) {
    final Map<EddystoneUid, EddystoneResolveRequest> requestsRegister = new HashMap<>();
    for (EddystoneResolveRequest request : requests) {
      EddystoneUid eddystoneUID = EddystoneUid.fromDevice(request.getFakeDevice());
      requestsRegister.put(eddystoneUID, request);
    }
    return requestsRegister;
  }

  private List<EddystoneFutureUID> resolveShuffles(final Map<EddystoneUid, EddystoneResolveRequest> requestsRegister) throws Exception {
    final List<EddystoneFutureUID> shuffles = new ArrayList<>();
    for (EddystoneResolveStrategy strategy : strategies) {
      List<EddystoneFutureUID> resolved = strategy.resolve(requestsRegister);
      shuffles.addAll(resolved);
    }
    return shuffles;
  }

  private Map<EddystoneUid, EddystoneFutureUID> buildShufflesRegister(final List<EddystoneFutureUID> shuffles) {
    final Map<EddystoneUid, EddystoneFutureUID> shufflesRegister = new HashMap<>();
    for (EddystoneFutureUID futureShuffle : shuffles) {
      shufflesRegister.put(futureShuffle.getQueriedBy(), futureShuffle);
    }
    return shufflesRegister;
  }

  private void evictOutdatedCacheEntries(final Map<EddystoneUid, EddystoneResolveRequest> requestsRegister,
      final Map<EddystoneUid, EddystoneFutureUID> shufflesRegister) {
    final List<String> uniqueIds = new ArrayList<>();
    for (Map.Entry<EddystoneUid, EddystoneResolveRequest> entry : requestsRegister.entrySet()) {
      EddystoneUid queriedBy = entry.getKey();
      EddystoneResolveRequest request = entry.getValue();
      EddystoneFutureUID eddystoneFutureUID = shufflesRegister.get(queriedBy);
      if (eddystoneFutureUID == null || ResolverType.CACHE == request.getResolvedBy()) {
        continue;
      }
      // non null shuffles resolved by api
      uniqueIds.add(eddystoneFutureUID.getUniqueId());
    }

    // remove old (OLD_DEVICE_ID -> RESOLVED_ID) entries from cache and
    // replace them with the new ones (NEW_DEVICE_ID -> RESOLVED_ID) in next step (@addNewCacheEntries)
    cache.evict(uniqueIds, DEVICE_PROFILE);
  }

  private void handleRequests(final Map<EddystoneUid, EddystoneResolveRequest> requestsRegister,
      final Map<EddystoneUid, EddystoneFutureUID> shufflesRegister) {
    for (Map.Entry<EddystoneUid, EddystoneResolveRequest> entry : requestsRegister.entrySet()) {
      handleRequest(shufflesRegister, entry);
    }
  }

  private void handleRequest(Map<EddystoneUid, EddystoneFutureUID> shufflesRegister, Map.Entry<EddystoneUid, EddystoneResolveRequest> entry) {
    final EddystoneUid queriedBy = entry.getKey();
    final EddystoneFutureUID eddystoneFutureUID = shufflesRegister.get(queriedBy);
    if (eddystoneFutureUID == null) {
      cache.populate(queriedBy.toString(), FutureShufflesCache.PHANTOM_ENTRY);
      return;
    }

    final EddystoneUid resolvedId = eddystoneFutureUID.getResolved();
    final EddystoneResolveRequest request = entry.getValue();

    final String uniqueId = eddystoneFutureUID.getUniqueId();
    final ResolvedId resolvedEddystoneUID = ResolvedId.create(resolvedId.toString(), uniqueId, DEVICE_PROFILE);

    // after remove old entries from cache (@evictOutdatedCacheEntries), add the new ones
    if (ResolverType.CACHE != request.getResolvedBy()) {
      addNewCacheEntries(eddystoneFutureUID, resolvedEddystoneUID);
    }

    if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
      notifyListeners(request, resolvedEddystoneUID);
    }
  }

  private void addNewCacheEntries(final EddystoneFutureUID eddystoneFutureUID, final ResolvedId resolvedEddystoneUID) {
    for (EddystoneUid futureShuffle : eddystoneFutureUID.getFutureIds()) {
      cache.populate(futureShuffle.toString(), resolvedEddystoneUID);
    }
    cache.populate(resolvedEddystoneUID.getEddystoneUID().toString(), resolvedEddystoneUID);
  }

  private void notifyListeners(final EddystoneResolveRequest request, final ResolvedId resolvedEddystoneUID) {
    final IEddystoneDevice fakeDevice = request.getFakeDevice();
    final IEddystoneDevice resolvedDevice = new EddystoneDevice.Builder(fakeDevice).resolvedId(resolvedEddystoneUID).build();
    cache.notifyListeners(resolvedDevice);
  }
}
