package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.bluetooth.BluetoothDevice;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver;
import com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver;
import com.kontakt.sdk.android.ble.spec.KontaktTelemetry;
import com.kontakt.sdk.android.common.model.Model;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.util.ConversionUtils;

import java.util.Arrays;

import static com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver.SECURE_PROFILE_INSTANCE_ID_START_INDEX;
import static com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver.SECURE_PROFILE_NAMESPACE_ID_START_INDEX;

@SuppressWarnings("MissingPermission")
final class SecureProfileParser extends Parser<SecureProfile> {

  private static final byte[] KONTAKT_SECURE_PROFILE_PREFIX = new byte[]{0x16, 0x6A, (byte) 0xFE};
  private static final int MIN_SECURE_PROFILE_LENGTH = 14;
  private static final byte SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD = 0x01;
  private static final byte PLAIN_DEVICE_IDENTIFIER_PAYLOAD = 0x02;
  private static final byte KONTAKT_TELEMETRY_IDENTIFIER_PAYLOAD = 0x03;
  private static final NamespaceIdResolver NAMESPACE_RESOLVER = new NamespaceIdResolver(SECURE_PROFILE_NAMESPACE_ID_START_INDEX);
  private static final InstanceIdResolver INSTANCE_ID_RESOLVER = new InstanceIdResolver(SECURE_PROFILE_INSTANCE_ID_START_INDEX);
  private static final KontaktTLMResolver KONTAKT_TLM_RESOLVER = new KontaktTLMResolver();

  SecureProfileParser(final ScanContext scanContext) {
    super(scanContext);
  }

  boolean isValidSecureProfileFrame(final byte[] scanRecord) {
    return ConversionUtils.doesArrayContainSubset(scanRecord, KONTAKT_SECURE_PROFILE_PREFIX, 4) && scanRecord.length > MIN_SECURE_PROFILE_LENGTH;
  }

  void parseScanRecord(final byte[] scanRecord) {
    frameData.clear();
    extractFrameData(scanRecord, frameData);
  }

  ISecureProfile getSecureProfile(final BluetoothDevice device, int rssi) {
    final byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    final byte[] nameData = frameData.get(FrameDataType.LOCAL_NAME);

    // Parse common data
    final byte payloadId = serviceData[2];
    final boolean isShuffled = (payloadId == SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD);
    final String name = parseName(nameData);

    // Check if already exists in cache
    final SecureProfile cachedSecureDevice = getCachedSecureDevice(device.getAddress());
    final SecureProfile.Builder builder = cachedSecureDevice != null ?
        new SecureProfile.Builder(cachedSecureDevice) : new SecureProfile.Builder();

    // Update common data
    builder
        .macAddress(device.getAddress())
        .shuffled(isShuffled)
        .name(name);

    // Update specified data based on payload ID
    if (payloadId == PLAIN_DEVICE_IDENTIFIER_PAYLOAD) {
      parsePlainDevicePayload(serviceData, builder);
    }
    if (payloadId == SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD) {
      parseShuffledDevicePayload(serviceData, builder);
    }
    if (payloadId == KONTAKT_TELEMETRY_IDENTIFIER_PAYLOAD) {
      parseTelemetryPayload(serviceData, builder);
    }

    // Build secure profile and update rssi
    final SecureProfile secureProfile = builder.build();
    updateRssi(secureProfile, rssi);

    // Put updated device in cache
    putSecureDeviceInCache(device.getAddress(), secureProfile);

    return secureProfile;
  }

  private void parsePlainDevicePayload(final byte[] serviceData, final SecureProfile.Builder builder) {
    final Model model = Model.fromCode(serviceData[3]);
    final String firmware = parseFirmwareVersion(serviceData);
    final int batteryLevel = serviceData[6];
    final int txPower = serviceData[7];
    final String uniqueId = parseUniqueId(serviceData);

    builder
        .model(model)
        .firmwareRevision(firmware)
        .batteryLevel(batteryLevel)
        .txPower(txPower)
        .uniqueId(uniqueId);
  }

  private void parseShuffledDevicePayload(final byte[] serviceData, final SecureProfile.Builder builder) {
    final Model model = Model.fromCode(serviceData[3]);
    final String firmware = parseFirmwareVersion(serviceData);
    final int batteryLevel = serviceData[6];
    final int txPower = serviceData[7];
    String namespace = parseNamespace(serviceData);
    String shuffledInstanceId = parseShuffledInstanceId(serviceData);

    builder
        .model(model)
        .firmwareRevision(firmware)
        .batteryLevel(batteryLevel)
        .txPower(txPower)
        .namespace(namespace)
        .instanceId(shuffledInstanceId);
  }

  private void parseTelemetryPayload(final byte[] serviceData, final SecureProfile.Builder builder) {
    final KontaktTelemetry telemetry = KONTAKT_TLM_RESOLVER.parse(serviceData);
    builder.telemetry(telemetry);
  }

  private void putSecureDeviceInCache(String deviceAddress, SecureProfile device) {
    int hashCode = hashCodeBuilder.append(deviceAddress).build();
    devicesCache.put(hashCode, device);
  }

  SecureProfile getCachedSecureDevice(String deviceAddress) {
    int hashCode = hashCodeBuilder.append(deviceAddress).build();
    return devicesCache.get(hashCode);
  }

  private void updateRssi(SecureProfile device, int rssi) {
    int calculatedRssi = rssiCalculator.calculateRssi(device.getMacAddress().hashCode(), rssi);
    device.setRssi(calculatedRssi);
  }

  private String parseName(final byte[] nameData) {
    if (nameData == null) {
      return null;
    }
    return new String(nameData);
  }

  private String parseFirmwareVersion(final byte[] serviceData) {
    final int firmwareVersionMajor = serviceData[4];
    final int firmwareVersionMinor = serviceData[5];

    if (firmwareVersionMajor < 0 || firmwareVersionMinor < 0) {
      return "";
    }
    return String.format("%d.%d", firmwareVersionMajor, firmwareVersionMinor);
  }

  private String parseUniqueId(final byte[] serviceData) {
    return new String(Arrays.copyOfRange(serviceData, 8, serviceData.length));
  }

  private String parseNamespace(final byte[] serviceData) {
    return NAMESPACE_RESOLVER.parse(serviceData);
  }

  private String parseShuffledInstanceId(final byte[] serviceData) {
    return INSTANCE_ID_RESOLVER.parse(serviceData);
  }

  @Override
  protected void disable() {
    if (isEnabled) {
      isEnabled = false;
    }
  }
}
