package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.bluetooth.BluetoothDevice;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.BeaconDevice;
import com.kontakt.sdk.android.ble.discovery.DiscoveryUtils;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.util.Arrays;
import java.util.Set;
import java.util.UUID;

@SuppressWarnings("MissingPermission")
public final class IBeaconParser extends Parser<BeaconDevice> {

  public static final byte[] MANUFACTURER_DATA_IBEACON_PREFIX = new byte[] { 0x4C, 0x00, 0x02, 0x15 };
  private static final int MANUFACTURER_DATA_IBEACON_LENGTH = 25;

  private final Set<IBeaconRegion> regions;

  public IBeaconParser(final ScanContext scanContext) {
    super(scanContext);
    regions = scanContext.getIBeaconRegions();
  }

  public void parseScanRecord(final byte[] scanRecord) {
    frameData.clear();
    extractFrameData(scanRecord, frameData);
  }

  boolean isValidIBeaconFrame(final byte[] scanRecord) {
    return ConversionUtils.doesArrayContainSubset(scanRecord, MANUFACTURER_DATA_IBEACON_PREFIX, 5);
  }

  boolean isManufacturerDataValid() {
    final byte[] manufacturerData = frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA);
    return manufacturerData != null && manufacturerData.length == MANUFACTURER_DATA_IBEACON_LENGTH;
  }

  IBeaconDevice getIBeaconDevice(final BluetoothDevice device, final int rssi) {
    int deviceHashCode = hashCodeBuilder.append(device.getAddress()).append(frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA)).build();

    BeaconDevice iBeaconDevice = devicesCache.get(deviceHashCode);
    if (iBeaconDevice != null) {
      update(iBeaconDevice, deviceHashCode, rssi);
      return new BeaconDevice.Builder(iBeaconDevice).build();
    }

    byte[] manufacturerData = frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA);
    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);

    final ScanResponse scanResponse = ScanResponse.fromScanResponseBytes(serviceData);
    final UUID proximityUUID = ConversionUtils.toUUID(Arrays.copyOfRange(manufacturerData, 4, 20));
    final int major = ConversionUtils.asInt(Arrays.copyOfRange(manufacturerData, 20, 22));
    final int minor = ConversionUtils.asInt(Arrays.copyOfRange(manufacturerData, 22, 24));
    final int calculatedRssi = rssiCalculator.calculateRssi(device.getAddress().hashCode(), rssi);
    int txPower = manufacturerData[24];
    final double distance = DiscoveryUtils.calculateDistance(txPower, calculatedRssi, DeviceProfile.IBEACON);

    iBeaconDevice = new BeaconDevice.Builder()
        .address(device.getAddress())
        .uniqueId(scanResponse.getUniqueId())
        .batteryPower(scanResponse.getBatteryPower())
        .firmwareRevision(scanResponse.getFirmwareVersion())
        .shuffled(isShuffled(scanResponse, proximityUUID))
        .name(device.getName())
        .proximityUUID(proximityUUID)
        .major(major)
        .minor(minor)
        .distance(distance)
        .proximity(Proximity.fromDistance(distance))
        .txPower(txPower)
        .timestamp(System.currentTimeMillis())
        .rssi(calculatedRssi)
        .build();

    devicesCache.put(deviceHashCode, iBeaconDevice);

    return new BeaconDevice.Builder(iBeaconDevice).build();
  }

  private void update(BeaconDevice device, int deviceHashCode, int rssi) {
    int calculatedRssi = rssiCalculator.calculateRssi(deviceHashCode, rssi);
    int txPower = frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA)[24];
    double distance = DiscoveryUtils.calculateDistance(txPower, calculatedRssi, DeviceProfile.IBEACON);

    device.setDistance(distance);
    device.setRssi(calculatedRssi);
    device.setProximity(Proximity.fromDistance(distance));
    device.setTimestamp(System.currentTimeMillis());
  }

  private boolean isShuffled(ScanResponse scanResponse, UUID proximityUUID) {
    return scanResponse.isUnknown() ? isSecureRegionDefined(proximityUUID) : scanResponse.isShuffled();
  }

  private boolean isSecureRegionDefined(UUID proximityUUID) {
    for (IBeaconRegion region : regions) {
      if (region.getSecureProximity() != null && proximityUUID.equals(region.getSecureProximity())) {
        return true;
      }
    }
    return false;
  }

  @Override
  protected void disable() {
    if (isEnabled) {
      isEnabled = false;
      devicesCache.clear();
      rssiCalculator.clear();
    }
  }
}
