package com.kontakt.sdk.android.ble.connection;

import com.kontakt.sdk.android.ble.dfu.FirmwareUpdateListener;
import com.kontakt.sdk.android.ble.security.auth.AuthToken;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Credentials;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.model.Preset;
import com.kontakt.sdk.android.common.model.Time;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.util.UUID;

/**
 * KontaktDeviceConnection connection allows to establish connection with beacon.<br>
 * Since firmware in version 4.0 device does not to have setup password, but below it still needs set password
 * <br><br>
 * All operation with device should be as quick as possible. You should not open connection and wait long time before taking any action.
 * After doing job connection should be closed immediately
 */
public interface KontaktDeviceConnection {

  /**
   * Applies pending configuration ({@link Config}) for Device.
   * <br>
   * It is possible to encounter problems during batch (the Config in this case)
   * write operation which result in launching error handle callback method.
   * Once the write batch operation fails, the Write Batch Processor
   * will attempt performing rollback write batch operation aiming at
   * restoring original Beacon configuration.
   * The rollback operation however may not succeed as well leaving Beacon device
   * changed partially.
   * Therefore it is strongly advisable to write batches in relatively close distance
   * to Beacon device. Thus, the risk of leaving Beacon in unspecified state may
   * be avoided.
   * <br>
   * The method is synchronized.
   * <br><br>
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param config the config
   * @param writeBatchListener the write batch listener
   */
  void applyConfig(Config config, WriteBatchListener<Config> writeBatchListener);

  /**
   * Accepts predefined profile ({@link Preset}) and changes parameters for Beacon.
   * <br>
   * It is possible to encounter problems during batch (the Preset in this case) write operation which result
   * in launching error handle callback method. Once the write batch operation failes,
   * the Write Batch Processor will attempt performing rollback write batch
   * operation aiming at restoring original Beacon configuration.
   * The rollback operation however may not succeed as well leaving Beacon device
   * changed partially.
   * Therefore it is strongly advisable to write batches in relatively close distance
   * to Beacon device. Thus, the risk of leaving Beacon in unspecified state may
   * be avoided.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param profile the profile
   * @param writeBatchListener the write batch listener
   */
  void acceptProfile(Preset profile, WriteBatchListener<Preset> writeBatchListener);

  /**
   * Writes minor value. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param value the value
   * @param writeListener the write listener
   */
  void overwriteMinor(int value, WriteListener writeListener);

  /**
   * Changes Device mode to non-connectable. During the non-connectable mode
   * no connection can be established with the Device for about 20 minutes.
   * <br>
   * The non-connectable mode is also enabled when authentication failure
   * occured because of providing wrong password.
   * <br>
   * Device Master Password can be obtained via REST Client. The result
   * callback is invoked by WriteListener.
   * <br>
   * For more information concerning Device master password, see
   * Credentials model
   * ({@link Credentials})
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param masterPassword the beacon master password
   * @param writeListener the write listener
   */
  void enableNonConnectableMode(String masterPassword, WriteListener writeListener);

  /**
   * Changes Beacon major value. The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param value new value
   * @param writeListener the write listener
   */
  void overwriteMajor(int value, WriteListener writeListener);

  /**
   * Changes Beacon Proximity UUID value. The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param proximity the proximity UUID
   * @param writeListener the write listener
   */
  void overwriteProximityUUID(UUID proximity, WriteListener writeListener);

  /**
   * Changes beacon password. The password must be 4-byte long. For more
   * information see kontakt.io Datasheet - version 2.0.
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param newPassword the new password
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
   */
  void overwritePassword(String newPassword, WriteListener writeListener);

  /**
   * Changes Beacon model name. The name should be no longer than sequence of 15 bytes
   * (15 ASCII characters). For more information see kontakt.io
   * Datasheet - version 2.0.
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param newModelName the new model name
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
   */
  void overwriteModelName(String newModelName, WriteListener writeListener);

  /**
   * Changes Beacon advertising interval. The interval value is accepted in milliseconds and must
   * be within [20 - 10240] ms range.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param millis the interval millis
   * @param writeListener the write listener
   */
  void overwriteAdvertisingInterval(long millis, WriteListener writeListener);

  /**
   * Changes Beacon power level. The Power level is described by 8 levels where 0 is the
   * lowest (default) and 7 is the highest. For more information see
   * kontakt.io Datasheet - version 2.0.
   * <br>
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param powerLevel the power level
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  void overwritePowerLevel(int powerLevel, WriteListener writeListener);

  /**
   * Switch to device profile.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param deviceProfile the device profile
   * @param writeListener the write listener
   */
  void switchToDeviceProfile(DeviceProfile deviceProfile, WriteListener writeListener);

  /**
   * Resets device so that connection is established once again.
   * <br>
   * For more information see kontakt.io Datasheet - version 2.0.
   * <br>
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  void resetDevice(WriteListener writeListener);

  /**
   * Enables dfu mode. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param masterPassword the master password
   * @param writeListener the write listener
   */
  void enableDfuMode(String masterPassword, WriteListener writeListener);

  /**
   * Overwrites url. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param newUrl the new url
   * @param writeListener the write listener
   */
  void overwriteUrl(String newUrl, WriteListener writeListener);

  /**
   * Overwrites namespace id. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param namespaceId the namespace id
   * @param writeListener the write listener
   */
  void overwriteNamespaceId(String namespaceId, WriteListener writeListener);

  /**
   * Overwrite instance id. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param instanceId the instance id
   * @param writeListener the write listener
   */
  void overwriteInstanceId(String instanceId, WriteListener writeListener);

  /**
   * Restores default Beacon settings. As an argument, 6-byte long Beacon master password must be
   * provided. The master password can be obtained via API Client.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   *
   * @param masterPassword the master password
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  void restoreDefaultSettings(String masterPassword, WriteListener writeListener);

  /**
   * Authorizes with beacon.
   * Authorization is required before certain operations conducted on beacons with firmware 4.0 or higher and beacons pro.<br>
   * These operations are:<br>
   * - {@link #readTime(ReadListener)}<br>
   * - {@link #readLightSensor(ReadListener)}
   *
   * @param token {@link AuthToken} instance required for authorization.
   * @param callback Authorization result callback.
   */
  void authorize(AuthToken token, AuthorizationCallback callback);

  /**
   * Operation that returns beacon's current configuration wrapped in a {@link Config} object.<br>
   * Successful read all operation is equal to successful authorization operation {@link #authorize(AuthToken,
   * AuthorizationCallback)}.<br><br>
   * Only available for beacons with firmware 4.0 or higher and beacons pro.<br>
   *
   * @param token {@link AuthToken} instance.
   * @param listener Read listener.
   */
  void readAll(AuthToken token, ReadListener<Config> listener);

  /**
   * Operation that returns current UTC time on a device.<br>
   * Only available for beacons pro.<br>
   * Authorization is required before this operation. See {@link #authorize(AuthToken, AuthorizationCallback)}
   *
   * @param listener Read listener
   */
  void readTime(ReadListener<Time> listener);

  /**
   * Operation that returns current light sensor reading on a device. Unit: 0 to 100 %<br>
   * Only available for Beacons Pro with firmware 1.7 or higher.<br>
   * Reading light sensor is based on characteristic notification so after using this method a sensor reading will be received continuously every
   * 0.5 of a second.
   * Authorization is required before this operation. See {@link #authorize(AuthToken, AuthorizationCallback)}
   *
   * @param listener Read listener
   */
  void readLightSensor(ReadListener<Integer> listener);

  /**
   * Method for writing secure config. Accessible for device with firmware 4.0 and above and Beacon Pro
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided device's firmware does not support this operation
   * *
   */
  void applySecureConfig(String encryptedConfig, WriteListener writeListener);

  /**
   * Method for writing secure config offline. Accessible for devices with firmware 4.0 or higher and Beacon Pro
   * <br>
   * WARNING: This is an advanced feature that should not be used in normal situations.
   * By applying offline secure configs you accept the fact that the beacon might be desynchronized with Kontakt Cloud.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided device's firmware does not support this operation
   *
   * @param config Config that is about to be applied.
   * @param token {@link AuthToken} token.
   * @param writeListener Operation listener
   */
  void applySecureConfig(Config config, AuthToken token, WriteListener writeListener);

  /**
   * Executes Secure Command.
   *
   * @param secureCommand String representing secure command.
   * @param writeListener Execution listener.
   */
  void executeSecureCommand(String secureCommand, WriteListener writeListener);

  /**
   * Method for initializing firmware update for Kontakt.io devices with secure profile (for now: Beacon PRO only)
   *
   * @param firmware {@link Firmware} instance.
   * @param kontaktCloud {@link KontaktCloud} instance. This is required for Beacon authorization and downloading firmware file.
   * @param firmwareUpdateListener {@link FirmwareUpdateListener} instance. Listener that will report firmware update progress and errors.
   */
  void updateFirmware(Firmware firmware, KontaktCloud kontaktCloud, FirmwareUpdateListener firmwareUpdateListener);

  /**
   * Method for initializing firmware update for Kontakt.io devices with secure profile (for now: Beacon PRO only).
   * This method should be used only uf you have pre-loaded byte array of valid firmware file.
   *
   * @param firmware {@link Firmware} instance.
   * @param fileBytes Firmware file bytes.
   * @param kontaktCloud {@link KontaktCloud} instance. This is required for Beacon authorization.
   * @param firmwareUpdateListener {@link FirmwareUpdateListener} instance. Listener that will report firmware update progress and errors.
   */
  void updateFirmware(Firmware firmware, byte[] fileBytes, KontaktCloud kontaktCloud, FirmwareUpdateListener firmwareUpdateListener);

  /**
   * Operation that synchronizes current UTC time with the beacon.<br>
   * Only available for beacons pro.<br>
   * Authorization is required before this operation. See {@link #authorize(AuthToken, AuthorizationCallback)}
   *
   * @param writeListener Write listener (extra will always be null in this case)
   */
  void syncTime(WriteListener writeListener);

  /**
   * Gets beacon that the connection is established with.
   *
   * @return the iBeacon device.
   */
  RemoteBluetoothDevice getDevice();

  /**
   * Initializes connection to device provided in the KontaktDeviceConnection constructor.
   *
   * @return true, if the connection attempt was initiated successfully
   */
  boolean connect();

  /**
   * Initializes connection to given device.
   *
   * @param remoteBluetoothDevice device to connect to.
   * @return true, if the connection attempt was initiated successfully.
   */
  boolean connect(RemoteBluetoothDevice remoteBluetoothDevice);

  /**
   * Closes Beacon connection releases provided resources.
   * The method is synchronized.
   */
  void close();

  /**
   * Returns true if Connection with Beacon is established.
   * The method is synchronized.
   *
   * @return the boolean flag indicating whether connection is established.
   */
  boolean isConnected();

  /**
   * Returns true if Beacon was successfully authenticated.
   * The method is synchronized.
   *
   * @return the boolean flag indicating whether beacon was authenticated.
   */
  boolean isAuthenticated();

  /**
   * Returns true if connection is closed.
   *
   * @return the boolean flag indicating whether connection is closed.
   */
  boolean isClosed();

  interface ConnectionListener {
    /**
     * Called when Android Device established connection with beacon.
     */
    void onConnectionOpened();

    /**
     * Called when Beacon device was successfully authenticated and returned
     * requested characteristics
     * ({@link com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice.Characteristics})
     *
     * @param characteristics beacon characteristics
     */
    void onAuthenticationSuccess(RemoteBluetoothDevice.Characteristics characteristics);

    /**
     * Called when the authorization password sent to beacon is incorrect.
     * After unsuccessful authorization Beacon will change its state to
     * Non-connectable mode. During this mode(for about 20 minutes),
     * obtaining Beacon device characteristics will be impossible.
     *
     * @param failureCode the failure code indicating what caused authentication
     * failure.
     */
    void onAuthenticationFailure(int failureCode);

    /**
     * Called when Beacon any characteristic was succesfully changed.
     *
     * @param characteristics the characteristics
     */
    void onCharacteristicsUpdated(RemoteBluetoothDevice.Characteristics characteristics);

    /**
     * Called when unexpected error occured during interacting with Beacon device.
     *
     * @param errorCode the error code
     */
    void onErrorOccured(int errorCode);

    /**
     * Called when Android device is disconnected from remote device.
     */
    void onDisconnected();
  }
}
