package com.kontakt.sdk.android.ble.service;

import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.common.log.Logger;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

@SuppressWarnings("WeakerAccess")
class ScanController {

  static final ScanController NULL = new ScanController(new Builder()) {
    @Override
    void start() {
      //Noop
    }

    @Override
    void stop() {
      //Noop
    }
  };

  private static final int MESSAGE_DISMISS = -1;
  private static final int MESSAGE_MONITOR_START = 0;
  private static final int MESSAGE_MONITOR_STOP = 1;

  private final ScanPeriod scanPeriod;
  private final BlockingQueue<Integer> messageQueue;
  private final Runnable monitorActiveRunner;
  private final Runnable monitorPassiveRunner;
  private final ForceScanScheduler forceScanScheduler;

  private Thread producerThread;
  private Thread consumerThread;

  ScanController(Builder builder) {
    this.scanPeriod = builder.scanPeriod;
    this.monitorActiveRunner = builder.monitorActiveRunner;
    this.monitorPassiveRunner = builder.monitorPassiveRunner;
    this.forceScanScheduler = builder.forceScanScheduler;
    this.messageQueue = new LinkedBlockingQueue<>(1);
  }

  void start() {
    producerThread = new Producer(messageQueue, scanPeriod, "monitor-message-producer-thread");
    consumerThread = new Consumer(messageQueue, monitorActiveRunner, monitorPassiveRunner, forceScanScheduler, "monitor-message-consumer-thread");
    consumerThread.start();
    producerThread.start();
  }

  void stop() {
    if (producerThread != null) {
      producerThread.interrupt();
      producerThread = null;
      consumerThread = null;
    }
  }

  private static class Producer extends Thread implements Thread.UncaughtExceptionHandler {

    private final BlockingQueue<Integer> messageQueue;
    private final long monitorActivePeriod;
    private final long monitorPassivePeriod;
    private final boolean isInstantScanRequested;

    Producer(BlockingQueue<Integer> messageQueue, ScanPeriod scanPeriod, String name) {
      super(name);
      this.messageQueue = messageQueue;
      this.monitorActivePeriod = scanPeriod.getActivePeriod();
      this.monitorPassivePeriod = scanPeriod.getPassivePeriod();

      this.isInstantScanRequested = monitorPassivePeriod == 0L;

      setUncaughtExceptionHandler(this);
    }

    @Override
    public void run() {
      try {
        boolean isFirstTimeScan = true;

        while (!Thread.currentThread().isInterrupted()) {
          Logger.d("Starting monitoring");

          if (isFirstTimeScan) {
            messageQueue.put(MESSAGE_MONITOR_START);
          } else if (!isInstantScanRequested) {
            messageQueue.put(MESSAGE_MONITOR_START);
          }

          TimeUnit.MILLISECONDS.sleep(monitorActivePeriod);
          Logger.d(": Stopping monitoring");

          if (!isInstantScanRequested) {
            messageQueue.put(MESSAGE_MONITOR_STOP);
          }

          TimeUnit.MILLISECONDS.sleep(monitorPassivePeriod);

          isFirstTimeScan = false;
        }
        Logger.d("Dismissing consumer thread");

        if (isInstantScanRequested) {
          messageQueue.put(MESSAGE_MONITOR_STOP);
        }

        messageQueue.put(MESSAGE_DISMISS);
      } catch (InterruptedException ignored) {
        try {
          if (isInstantScanRequested) {
            messageQueue.put(MESSAGE_MONITOR_STOP);
          }

          messageQueue.put(MESSAGE_DISMISS);
        } catch (InterruptedException e) {
          Logger.d("Monitoring interrupted");
        }
      }
    }

    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
      Logger.e(String.format("%s interrupted, exception swallowed.", thread.getName()));
    }
  }

  private static class Consumer extends Thread implements Thread.UncaughtExceptionHandler {

    private final BlockingQueue<Integer> messageQueue;
    private final Runnable monitorActiveRunner;
    private final Runnable monitorPassiveRunner;
    private final ForceScanScheduler forceScanScheduler;

    Consumer(BlockingQueue<Integer> messageQueue, Runnable monitorActiveRunner, Runnable monitorPassiveRunner, ForceScanScheduler forceScanScheduler,
        String name) {
      super(name);
      this.messageQueue = messageQueue;
      this.monitorActiveRunner = monitorActiveRunner;
      this.monitorPassiveRunner = monitorPassiveRunner;
      this.forceScanScheduler = forceScanScheduler;

      setUncaughtExceptionHandler(this);
    }

    @Override
    public void run() {
      try {
        while (true) {
          final int messageCode = messageQueue.take();
          switch (messageCode) {
            case MESSAGE_MONITOR_START:
              monitorActiveRunner.run();
              forceScanScheduler.start();
              break;
            case MESSAGE_MONITOR_STOP:
              forceScanScheduler.stop();
              monitorPassiveRunner.run();
              break;
            default:
              throw new InterruptedException();
          }
        }
      } catch (InterruptedException e) {
        Logger.d("Consumer thread interrupted");
      }
    }

    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
      Logger.e(String.format("%s interrupted, exception swallowed.", thread.getName()));
    }
  }

  static class Builder {
    Runnable monitorActiveRunner;
    Runnable monitorPassiveRunner;
    ScanPeriod scanPeriod;
    ForceScanScheduler forceScanScheduler;

    Builder setScanActiveRunner(Runnable monitorActiveRunner) {
      this.monitorActiveRunner = monitorActiveRunner;
      return this;
    }

    Builder setScanPassiveRunner(Runnable monitorPassiveRunner) {
      this.monitorPassiveRunner = monitorPassiveRunner;
      return this;
    }

    Builder setScanPeriod(ScanPeriod scanPeriod) {
      this.scanPeriod = scanPeriod;
      return this;
    }

    Builder setForceScanScheduler(ForceScanScheduler forceScanScheduler) {
      this.forceScanScheduler = forceScanScheduler;
      return this;
    }

    public ScanController build() {
      return new ScanController(this);
    }
  }
}
