package com.kontakt.sdk.android.ble.connection;

import android.content.Context;

import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.KontaktCloudFactory;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudError;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import java.util.UUID;

@SuppressWarnings("WeakerAccess")
class SyncableKontaktDeviceConnectionImpl implements SyncableKontaktDeviceConnection {

  /**
   * <p>
   * Abstract class used as listener to notify about updating device and syncing with REST API
   * <p>
   * <p>
   * Current implementation does nothing
   */
  public abstract static class SyncWriteListener {

    /**
     * The constant SYNC_NOT_IMPLEMENTED_YET.
     */
    public static final int SYNC_NOT_IMPLEMENTED_YET = -1;

    /**
     * Invoked when beacon characteristics manipulation failed.
     * <p>
     * If so no attempt will be done to update REST API
     *
     * @deprecated use {@link #onWriteFailed(ErrorCause)} instead
     */
    @Deprecated
    public void onWriteFailed() {
    }

    /**
     * Invoked when beacon characteristics manipulation failed.
     * <p>
     * If so no attempt will be done to update REST API
     *
     * @param cause {@link ErrorCause}
     */
    public void onWriteFailed(ErrorCause cause) {
    }

    /**
     * <p>
     * Invoked when beacon non secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e the exception object
     */
    public void onSyncFailed(KontaktCloudException e) {
    }

    /**
     * <p>
     * Invoked when beacon non secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e the error object
     */
    public void onSyncFailed(CloudError e) {
    }

    /**
     * <p>
     * Invoked when beacon beacon secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e        the exception object
     * @param response encrypted response from beacon
     */
    public void onSyncFailed(KontaktCloudException e, String response) {

    }

    /**
     * <p>
     * Invoked when beacon beacon secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e              the error object
     * @param secureResponse encrypted response from beacon
     */
    public void onSyncFailed(CloudError e, Config secureResponse) {

    }

    /**
     * <p>
     * Invoked when beacon characteristics manipulation was success and invoking REST API succeed
     */
    public void onSuccess() {
    }
  }

  /**
   * <p>
   * Abstract class used as listener to notify about batch updating device and syncing with REST API
   * <p>
   * <p>
   * Current implementation does nothing
   *
   * @param <T> Batch holder type
   */
  public abstract static class SyncWriteBatchListener<T> {

    /**
     * <p>
     * Informs that Android device starts writing parameters Batch to Beacon device.
     *
     * @param batchHolder the batch holder
     */
    public void onSyncWriteBatchStart(T batchHolder) {

    }

    /**
     * <p>
     * Informs that write failed
     */
    public void onWriteFailed() {

    }

    /**
     * Informs that an error has occured during batch write operation.
     * Once the callback method is launched the rollback operation is performed aiming at
     * restoring original configuration to Beacon device.
     *
     * @param errorCode the error code
     */
    public void onBatchWriteError(int errorCode) {

    }

    /**
     * <p>
     * Invoked when beacon batch characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e the exception object
     */
    public void onSyncFailed(KontaktCloudException e) {

    }

    /**
     * <p>
     * Invoked when beacon batch characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param message   the failure message
     * @param errorCode the error code
     */
    public void onSyncFailed(final String message, final int errorCode) {

    }

    /**
     * <p>
     * Invoked when beacon batch characteristics manipulation was success and invoking REST API succeed
     *
     * @param batchHolder the batch holder
     */
    public void onSuccess(T batchHolder) {

    }
  }

  private static final DeviceType DEVICE_TYPE = DeviceType.BEACON;

  private KontaktDeviceConnectionImpl kontaktDeviceConnection;
  private KontaktCloud kontaktCloud;

  /**
   * Instantiates a new Syncable Kontakt device connection.
   *
   * @param context               the context
   * @param remoteBluetoothDevice the beacon device
   * @param connectionListener    the connection listener
   */
  SyncableKontaktDeviceConnectionImpl(Context context, RemoteBluetoothDevice remoteBluetoothDevice,
                                      KontaktDeviceConnectionImpl.ConnectionListener connectionListener) {
    kontaktDeviceConnection = new KontaktDeviceConnectionImpl(context, remoteBluetoothDevice, connectionListener);
    kontaktCloud = KontaktCloudFactory.create();
  }

  @Override
  public synchronized boolean connectToDevice() {
    return kontaktDeviceConnection.connect();
  }

  @Override
  public synchronized boolean isConnectedToDevice() {
    return kontaktDeviceConnection.isConnected();
  }

  @Override
  public synchronized void close() {
    kontaktDeviceConnection.close();
  }

  @Override
  public synchronized RemoteBluetoothDevice getDevice() {
    return kontaktDeviceConnection.getDevice();
  }

  @Override
  public synchronized boolean isDeviceAuthenticated() {
    return kontaktDeviceConnection.isAuthenticated();
  }

  @Override
  public synchronized void overwriteMinor(final int value, final SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().minor(value).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteMinor(value, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwriteMajor(int value, final SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().major(value).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteMajor(value, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwritePowerLevel(int value, final SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().txPower(value).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwritePowerLevel(value, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwriteProximityUUID(UUID proximityUUID, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().proximity(proximityUUID).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteProximityUUID(proximityUUID, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwritePassword(String password, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().password(password).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwritePassword(password, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwriteAdvertisingInterval(long interval, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().interval((int) interval).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteAdvertisingInterval(interval, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwriteModelName(String modelName, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().name(modelName).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteModelName(modelName, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void resetDevice(WriteListener writeListener) {
    kontaktDeviceConnection.resetDevice(writeListener);
  }

  @Override
  public void overwriteUrl(String url, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().url(url).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteUrl(url, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwriteNamespace(String namespaceId, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().namespace(namespaceId).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteNamespaceId(namespaceId, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void overwriteInstanceId(String instanceId, SyncWriteListener syncWriteListener) {
    final Config config = new Config.Builder().instanceId(instanceId).build();
    final Device device = new Device.Builder().uniqueId(getDeviceUniqueId()).deviceType(DEVICE_TYPE).config(config).build();
    kontaktDeviceConnection.overwriteInstanceId(instanceId, new InternalBeaconWriteListener(kontaktCloud, syncWriteListener, device));
  }

  @Override
  public void applyConfig(Config config, SyncWriteBatchListener<Config> syncWriteBatchListener) {
    kontaktDeviceConnection.applyConfig(config, new InternalBeaconWriteBatchListener<>(kontaktCloud, syncWriteBatchListener));
  }

  @Override
  public void applySecureConfig(Config secureConfig, SyncWriteListener syncWriteListener) {
    kontaktDeviceConnection.applySecureConfig(secureConfig.getSecureRequest(),
        new InternalSecureBeaconWriteListener(syncWriteListener, kontaktCloud, getDeviceUniqueId()));
  }

  private String getDeviceUniqueId() {
    return getDevice().getUniqueId();
  }
}
