package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * This class represents shuffled SecureProfile-UID composed of 10-byte namespace and a 6-byte instance ID.
 */
public class SecureProfileUid implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<SecureProfileUid> CREATOR = new Creator<SecureProfileUid>() {
    @Override
    public SecureProfileUid createFromParcel(Parcel source) {
      return new SecureProfileUid(source);
    }

    @Override
    public SecureProfileUid[] newArray(int size) {
      return new SecureProfileUid[size];
    }
  };

  private final String namespace;
  private final String instanceId;

  /**
   * Creates new SecureProfile-UID object with specified namespace and instance ID.
   *
   * @param namespace the namespace.
   * @param instanceId the instance ID.
   * @return the SecureProfile-UID instance.
   */
  public static SecureProfileUid of(String namespace, String instanceId) {
    return new SecureProfileUid(namespace, instanceId);
  }

  /**
   * Creates new SecureProfile-UID object from {@link ISecureProfile} instance.
   *
   * @param fakeDevice the SecureProfile device instance.
   * @return the SecureProfile-UID instance.
   */
  public static SecureProfileUid fromDevice(ISecureProfile fakeDevice) {
    SDKPreconditions.checkNotNull(fakeDevice, "device cannot be null");
    return new SecureProfileUid(fakeDevice.getNamespace(), fakeDevice.getInstanceId());
  }

  /**
   * Creates new SecureProfile-UID object from {@link ISecureProfile} instance.
   *
   * @param eddystoneUid the SecureProfile device instance.
   * @return the SecureProfile-UID instance.
   */
  public static SecureProfileUid fromEddystoneUid(EddystoneUid eddystoneUid) {
    SDKPreconditions.checkNotNull(eddystoneUid, "EddystoneUid cannot be null");
    return new SecureProfileUid(eddystoneUid.getNamespace(), eddystoneUid.getInstanceId());
  }

  /**
   * Creates new SecureProfile-UID object from String value.
   *
   * @param queriedBy the string value in namespace:instanceID format.
   * @return the SecureProfile-UID instance.
   * @throws NullPointerException if input String is null.
   * @throws IllegalArgumentException if input String has invalid format.
   */
  public static SecureProfileUid fromQueriedBy(String queriedBy) {
    SDKPreconditions.checkNotNull(queriedBy, "queriedBy is null");
    String[] split = queriedBy.split(":");
    if (split.length != 2) {
      throw new IllegalArgumentException("queriedBy has invalid format");
    }
    String namespace = split[0];
    String instanceId = split[1];
    return new SecureProfileUid(namespace, instanceId);
  }

  private SecureProfileUid(String namespace, String instanceId) {
    this.namespace = namespace;
    this.instanceId = instanceId;
  }

  protected SecureProfileUid(Parcel in) {
    this.namespace = in.readString();
    this.instanceId = in.readString();
  }

  public String getNamespace() {
    return namespace;
  }

  public String getInstanceId() {
    return instanceId;
  }

  // do not change implementation of toString() method, it is used as http request param as well
  @Override
  public String toString() {
    return namespace + ":" + instanceId;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof SecureProfileUid)) {
      return false;
    }
    if (o == this) {
      return true;
    }

    SecureProfileUid eddystoneUid = (SecureProfileUid) o;
    return SDKEqualsBuilder.start().equals(namespace, eddystoneUid.namespace).equals(instanceId, eddystoneUid.instanceId).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(namespace).append(instanceId).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.namespace);
    dest.writeString(this.instanceId);
  }
}
