package com.kontakt.sdk.android.cloud.adapter;

import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.common.model.Access;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.Model;
import com.kontakt.sdk.android.common.model.PacketType;
import com.kontakt.sdk.android.common.model.Shuffles;
import com.kontakt.sdk.android.common.model.Specification;
import com.kontakt.sdk.android.common.model.SubscriptionPlan;
import com.kontakt.sdk.android.common.model.Venue;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Converts {@link Device} objects to and from JSON.
 */
public class DeviceTypeAdapter extends BaseTypeAdapter<Device> {

  private ConfigTypeAdapter configTypeAdapter = new ConfigTypeAdapter();

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(JsonWriter out, Device device) throws IOException {
    out = out.beginObject();

    writeString(out, CloudConstants.Common.UNIQUE_ID_PARAMETER, device.getUniqueId());
    writeUUID(out, CloudConstants.Common.ID_PARAMETER, device.getId());
    writeFutureId(out, device.getShuffles());
    writeString(out, CloudConstants.Devices.MAC_PARAMETER, device.getMac());
    writeUUID(out, CloudConstants.Devices.SECURE_PROXIMITY_PARAMETER, device.getSecureProximity());
    writeString(out, CloudConstants.Devices.SECURE_NAMESPACE_PARAMETER, device.getSecureNamespace());
    writeConfig(out, device.getConfig());
    writeString(out, CloudConstants.Devices.FIRMWARE_PARAMETER, device.getFirmware());
    writeString(out, CloudConstants.Devices.ALIAS_PARAMETER, device.getAlias());
    writeTags(out, device.getTags());
    writeEnum(out, CloudConstants.Devices.DEVICE_TYPE_PARAMETER, device.getDeviceType());
    writeEnum(out, CloudConstants.Devices.SPECIFICATION_PARAMETER, device.getSpecification());
    writeEnum(out, CloudConstants.Devices.MODEL_PARAMETER, device.getModel());
    writeUUID(out, CloudConstants.Devices.MANAGER_ID_PARAMETER, device.getManagerId());
    writeVenue(out, device.getVenue());
    writeInteger(out, CloudConstants.Devices.ACTIONS_COUNT_PARAMETER, device.getActionsCount());
    writeEnum(out, CloudConstants.Common.ACCESS_PARAMETER, device.getAccess());
    writeString(out, CloudConstants.Common.LAT_PARAMETER, device.getLatitude());
    writeString(out, CloudConstants.Common.LAT_PARAMETER, device.getLongitude());
    writeEnum(out, CloudConstants.Devices.SUBSCRIPTION_PLAN, device.getSubscriptionPlan());

    out.endObject();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Device read(JsonReader in) throws IOException {
    final Device.Builder deviceBuilder = new Device.Builder();
    final Config.Builder configBuilder = new Config.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Common.UNIQUE_ID_PARAMETER:
          final String uniqueId = in.nextString();
          deviceBuilder.uniqueId(uniqueId);
          configBuilder.uniqueId(uniqueId);
          break;
        case CloudConstants.Common.ID_PARAMETER:
          final UUID id = UUID.fromString(in.nextString());
          deviceBuilder.id(id);
          break;
        case CloudConstants.Devices.VENUE_PARAMETER:
          final Venue venue = readVenue(in);
          deviceBuilder.venue(venue);
          break;
        case CloudConstants.Devices.FUTURE_ID_PARAMETER:
          final Shuffles shuffles = readShuffles(in);
          deviceBuilder.shuffles(shuffles);
          break;
        case CloudConstants.Devices.MAC_PARAMETER:
          final String mac = in.nextString();
          deviceBuilder.mac(mac);
          break;
        case CloudConstants.Devices.SECURE_PROXIMITY_PARAMETER:
          final UUID secureProximity = UUID.fromString(in.nextString());
          deviceBuilder.secureProximity(secureProximity);
          break;
        case CloudConstants.Devices.SECURE_NAMESPACE_PARAMETER:
          final String secureNamespace = in.nextString();
          deviceBuilder.secureNamespace(secureNamespace);
          break;
        case CloudConstants.Devices.FIRMWARE_PARAMETER:
          final String firmware = in.nextString();
          deviceBuilder.firmware(firmware);
          break;
        case CloudConstants.Devices.ALIAS_PARAMETER:
          final String alias = in.nextString();
          deviceBuilder.alias(alias);
          break;
        case CloudConstants.Devices.TAGS_PARAMETER:
          final List<String> tags = readTags(in);
          deviceBuilder.tags(tags);
          break;
        case CloudConstants.Devices.DEVICE_TYPE_PARAMETER:
          final DeviceType deviceType = DeviceType.valueOf(in.nextString());
          deviceBuilder.deviceType(deviceType);
          break;
        case CloudConstants.Devices.SPECIFICATION_PARAMETER:
          final Specification specification = Specification.valueOf(in.nextString());
          deviceBuilder.specification(specification);
          break;
        case CloudConstants.Devices.MODEL_PARAMETER:
          final Model model = Model.valueOf(in.nextString());
          deviceBuilder.model(model);
          break;
        case CloudConstants.Devices.SUBSCRIPTION_PLAN:
          final SubscriptionPlan plan = SubscriptionPlan.fromString(in.nextString());
          deviceBuilder.subscriptionPlan(plan);
          break;
        case CloudConstants.Devices.MANAGER_ID_PARAMETER:
          final UUID managerId = UUID.fromString(in.nextString());
          deviceBuilder.managerId(managerId);
          break;
        case CloudConstants.Devices.ACTIONS_COUNT_PARAMETER:
          final int actionsCount = in.nextInt();
          deviceBuilder.actionsCount(actionsCount);
          break;
        case CloudConstants.Common.ACCESS_PARAMETER:
          final Access access = Access.valueOf(in.nextString());
          deviceBuilder.access(access);
          break;
        case CloudConstants.Common.LAT_PARAMETER:
          final String lat = in.nextString();
          deviceBuilder.latitude(lat);
          break;
        case CloudConstants.Common.LNG_PARAMETER:
          final String lng = in.nextString();
          deviceBuilder.longitude(lng);
          break;
        case CloudConstants.Devices.QUERIED_BY_PARAMETER:
          final String queriedBy = in.nextString();
          deviceBuilder.queriedBy(queriedBy);
          break;
        case CloudConstants.Configs.PROXIMITY_PARAMETER:
          final UUID proximity = UUID.fromString(in.nextString());
          configBuilder.proximity(proximity);
          break;
        case CloudConstants.Configs.MAJOR_PARAMETER:
          final int major = in.nextInt();
          configBuilder.major(major);
          break;
        case CloudConstants.Configs.MINOR_PARAMETER:
          final int minor = in.nextInt();
          configBuilder.minor(minor);
          break;
        case CloudConstants.Configs.TX_POWER_PARAMETER:
          final int txPower = in.nextInt();
          configBuilder.txPower(txPower);
          break;
        case CloudConstants.Configs.INTERVAL_PARAMETER:
          final int interval = in.nextInt();
          configBuilder.interval(interval);
          break;
        case CloudConstants.Configs.NAMESPACE_PARAMETER:
          final String namespace = in.nextString();
          configBuilder.namespace(namespace);
          break;
        case CloudConstants.Configs.URL_PARAMETER:
          final String url = in.nextString();
          configBuilder.url(url);
          break;
        case CloudConstants.Configs.INSTANCE_ID_PARAMETER:
          final String instanceId = in.nextString();
          configBuilder.instanceId(instanceId);
          break;
        case CloudConstants.Configs.PROFILES_PARAMETER:
          final List<DeviceProfile> profiles = configTypeAdapter.readProfiles(in);
          configBuilder.profiles(profiles);
          break;
        case CloudConstants.Configs.PACKETS_PARAMETER:
          final List<PacketType> packets = configTypeAdapter.readPackets(in);
          configBuilder.packets(packets);
          break;
        case CloudConstants.Configs.SHUFFLED_PARAMETER:
          final boolean shuffled = in.nextBoolean();
          configBuilder.shuffled(shuffled);
          break;
        case CloudConstants.Configs.NAME_PARAMETER:
          final String name = in.nextString();
          configBuilder.name(name);
          break;
        case CloudConstants.Configs.PASSWORD_PARAMETER:
          final String password = in.nextString();
          configBuilder.password(password);
          break;
        case CloudConstants.Devices.POWER_SAVING:
          configBuilder.powerSaving(configTypeAdapter.readPowerSaving(in));
          break;
        case CloudConstants.Configs.RSSI_1M_PARAMETER:
          configBuilder.rssi1m(configTypeAdapter.readRssiCalibration(in));
          break;
        case CloudConstants.Configs.RSSI_0M_PARAMETER:
          configBuilder.rssi0m(configTypeAdapter.readRssiCalibration(in));
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    deviceBuilder.config(configBuilder.build());
    return deviceBuilder.build();
  }

  private void writeConfig(JsonWriter out, Config config) throws IOException {
    if (config == null) {
      return;
    }
    configTypeAdapter.write(out, config);
  }

  private void writeFutureId(JsonWriter out, Shuffles shuffles) throws IOException {
    out.name(CloudConstants.Devices.FUTURE_ID_PARAMETER);
    if (shuffles == null) {
      out.nullValue();
      return;
    }

    out.beginObject();
    writeEddystoneUids(out, shuffles.getEddystoneShuffles());
    writeIBeaconIds(out, shuffles.getIBeaconShuffles());
    out.endObject();
  }

  private void writeEddystoneUids(JsonWriter out, List<EddystoneUid> eddystoneUids) throws IOException {
    out.name(CloudConstants.Devices.EDDYSTONE_PARAMETER);
    if (eddystoneUids == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (EddystoneUid eddystoneUID : eddystoneUids) {
      writeEddystoneUid(out, eddystoneUID);
    }
    out.endArray();
  }

  private void writeEddystoneUid(JsonWriter out, EddystoneUid eddystoneUid) throws IOException {
    out.beginObject();
    writeString(out, CloudConstants.Devices.NAMESPACE_PARAMETER, eddystoneUid.getNamespace());
    writeString(out, CloudConstants.Devices.INSTANCE_ID_PARAMETER, eddystoneUid.getInstanceId());
    out.endObject();
  }

  private void writeIBeaconIds(JsonWriter out, List<IBeaconId> iBeaconIds) throws IOException {
    out.name(CloudConstants.Devices.IBEACON_PARAMETER);
    if (iBeaconIds == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (IBeaconId iBeaconid : iBeaconIds) {
      writeIBeaconId(out, iBeaconid);
    }
    out.endArray();
  }

  private void writeIBeaconId(JsonWriter out, IBeaconId iBeaconId) throws IOException {
    out.beginObject();
    writeUUID(out, CloudConstants.Devices.PROXIMITY_PARAMETER, iBeaconId.getProximity());
    writeInteger(out, CloudConstants.Devices.MAJOR_PARAMETER, iBeaconId.getMajor());
    writeInteger(out, CloudConstants.Devices.MINOR_PARAMETER, iBeaconId.getMinor());
    out.endObject();
  }

  private void writeVenue(JsonWriter out, Venue venue) throws IOException {
    out.name(CloudConstants.Devices.VENUE_PARAMETER);
    if (venue == null) {
      out.nullValue();
      return;
    }

    out.beginObject();
    writeString(out, CloudConstants.Venues.NAME_PARAMETER, venue.getName());
    writeString(out, CloudConstants.Venues.DESCRIPTION_PARAMETER, venue.getDescription());
    writeUUID(out, CloudConstants.Common.ID_PARAMETER, venue.getId());
    writeString(out, CloudConstants.Venues.COVER_TYPE_PARAMETER, venue.getCoverType());
    writeString(out, CloudConstants.Venues.IMAGE_PARAMETER, venue.getImageUrl());
    writeEnum(out, CloudConstants.Common.ACCESS_PARAMETER, venue.getAccess());
    writeInteger(out, CloudConstants.Venues.DEVICES_COUNT_PARAMETER, venue.getDevicesCount());
    writeString(out, CloudConstants.Common.LAT_PARAMETER, venue.getLatitude());
    writeString(out, CloudConstants.Common.LNG_PARAMETER, venue.getLongitude());
    out.endObject();
  }

  private void writeTags(JsonWriter out, List<String> tags) throws IOException {
    out.name(CloudConstants.Devices.TAGS_PARAMETER);
    if (tags == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (String tag : tags) {
      out.value(tag);
    }
    out.endArray();
  }

  private Shuffles readShuffles(JsonReader in) throws IOException {
    final Shuffles.Builder builder = new Shuffles.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Devices.EDDYSTONE_PARAMETER:
          final List<EddystoneUid> eddystoneUIDs = readFutureEddystoneUids(in);
          builder.eddystoneShuffles(eddystoneUIDs);
          break;
        case CloudConstants.Devices.IBEACON_PARAMETER:
          final List<IBeaconId> iBeaconIds = readFutureIBeaconIds(in);
          builder.iBeaconShuffles(iBeaconIds);
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    return builder.build();
  }

  private List<EddystoneUid> readFutureEddystoneUids(JsonReader in) throws IOException {
    in.beginArray();
    List<EddystoneUid> eddystoneUids = new ArrayList<>();
    while (in.hasNext()) {
      EddystoneUid eddystoneUID = readEddystoneUid(in);
      eddystoneUids.add(eddystoneUID);
    }
    in.endArray();
    return eddystoneUids;
  }

  private EddystoneUid readEddystoneUid(JsonReader in) throws IOException {
    final EddystoneUid eddystoneUid;
    String namespace = null;
    String instanceId = null;

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Devices.NAMESPACE_PARAMETER:
          namespace = in.nextString();
          break;
        case CloudConstants.Devices.INSTANCE_ID_PARAMETER:
          instanceId = in.nextString();
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    eddystoneUid = EddystoneUid.of(namespace, instanceId);
    return eddystoneUid;
  }

  private List<IBeaconId> readFutureIBeaconIds(JsonReader in) throws IOException {
    in.beginArray();
    List<IBeaconId> iBeaconIds = new ArrayList<>();
    while (in.hasNext()) {
      IBeaconId iBeaconId = readIBeaconId(in);
      iBeaconIds.add(iBeaconId);
    }
    in.endArray();
    return iBeaconIds;
  }

  private IBeaconId readIBeaconId(JsonReader in) throws IOException {
    final IBeaconId iBeaconId;
    UUID proximity = null;
    int major = 0;
    int minor = 0;

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Devices.PROXIMITY_PARAMETER:
          proximity = UUID.fromString(in.nextString());
          break;
        case CloudConstants.Devices.MAJOR_PARAMETER:
          major = in.nextInt();
          break;
        case CloudConstants.Devices.MINOR_PARAMETER:
          minor = in.nextInt();
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    iBeaconId = IBeaconId.of(proximity, major, minor);
    return iBeaconId;
  }

  private Venue readVenue(JsonReader in) throws IOException {
    Venue.Builder venueBuilder = new Venue.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Common.ID_PARAMETER:
          final UUID id = UUID.fromString(in.nextString());
          venueBuilder.id(id);
          break;
        case CloudConstants.Venues.NAME_PARAMETER:
          final String name = in.nextString();
          venueBuilder.name(name);
          break;
        case CloudConstants.Venues.DESCRIPTION_PARAMETER:
          final String description = in.nextString();
          venueBuilder.description(description);
          break;
        case CloudConstants.Venues.COVER_TYPE_PARAMETER:
          final String coverType = in.nextString();
          venueBuilder.coverType(coverType);
          break;
        case CloudConstants.Venues.IMAGE_PARAMETER:
          final String imageUrl = in.nextString();
          venueBuilder.imageUrl(imageUrl);
          break;
        case CloudConstants.Common.ACCESS_PARAMETER:
          final Access access = Access.valueOf(in.nextString());
          venueBuilder.access(access);
          break;
        case CloudConstants.Venues.DEVICES_COUNT_PARAMETER:
          final int devicesCount = in.nextInt();
          venueBuilder.devicesCount(devicesCount);
          break;
        case CloudConstants.Common.LAT_PARAMETER:
          final String lat = in.nextString();
          venueBuilder.latitude(lat);
          break;
        case CloudConstants.Common.LNG_PARAMETER:
          final String lng = in.nextString();
          venueBuilder.longitude(lng);
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    return venueBuilder.build();
  }

  List<String> readTags(JsonReader in) throws IOException {
    in.beginArray();
    List<String> tags = new ArrayList<>();
    while (in.hasNext()) {
      tags.add(in.nextString());
    }
    in.endArray();
    return tags;
  }
}
