package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * Class representing a real beacon device.
 * Please, note that in the SDK there is also another Device class
 * ({@link Device}) which represents a JSON model
 * for a device and is accessible while interacting with the REST Client.
 */
public class BeaconDevice implements IBeaconDevice {

  /**
   * The Parcelable CREATOR constant.
   */
  public static final Creator<BeaconDevice> CREATOR = new Creator<BeaconDevice>() {
    public BeaconDevice createFromParcel(Parcel in) {
      return new BeaconDevice(in);
    }

    public BeaconDevice[] newArray(int size) {
      return new BeaconDevice[size];
    }
  };

  private static final String PARCEL_RSSI = "rssi";
  private static final String PARCEL_ADDRESS = "parcel_address";
  private static final String PARCEL_TIMESTAMP = "parcel_timestamp";
  private static final String PARCEL_NAME = "parcel_name";
  private static final String SHUFFLED = "shuffled";

  private final HashCodeBuilder hashCodeBuilder;

  private String address;
  private UUID proximityUUID;
  private int major;
  private int minor;
  private int txPower;
  private String name;
  private String uniqueId;
  private String firmwareVersion;
  private int batteryPower;
  private boolean shuffled;
  private int rssi;
  private double distance;
  private Proximity proximity;
  private long timestamp;
  private byte[] password;

  private BeaconDevice(IBeaconDevice beaconDevice, ResolvedId resolvedId) {
    IBeaconId iBeaconId = resolvedId.getIBeaconId();
    this.proximityUUID = iBeaconId.getProximity();
    this.major = iBeaconId.getMajor();
    this.minor = iBeaconId.getMinor();
    this.uniqueId = resolvedId.getUniqueId();
    this.txPower = beaconDevice.getTxPower();
    this.distance = beaconDevice.getDistance();
    this.timestamp = beaconDevice.getTimestamp();
    this.firmwareVersion = beaconDevice.getFirmwareVersion();
    this.batteryPower = beaconDevice.getBatteryPower();
    this.proximity = beaconDevice.getProximity();
    this.rssi = beaconDevice.getRssi();
    this.address = beaconDevice.getAddress();
    this.name = beaconDevice.getName();
    this.shuffled = beaconDevice.isShuffled();
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  BeaconDevice(Parcel input) {
    final Bundle bundle = input.readBundle(getClass().getClassLoader());
    this.rssi = bundle.getInt(PARCEL_RSSI);
    this.name = bundle.getString(PARCEL_NAME);
    this.address = bundle.getString(PARCEL_ADDRESS);
    this.proximityUUID = (UUID) bundle.getSerializable(Constants.Devices.PROXIMITY);
    this.major = bundle.getInt(Constants.Devices.MAJOR);
    this.minor = bundle.getInt(Constants.Devices.MINOR);
    this.txPower = bundle.getInt(Constants.Devices.TX_POWER);
    this.distance = bundle.getDouble(Constants.Devices.ACCURACY);
    this.password = bundle.getByteArray(Constants.Devices.PASSWORD);
    this.timestamp = bundle.getLong(PARCEL_TIMESTAMP);
    this.proximity = Proximity.fromDistance(distance);
    this.firmwareVersion = bundle.getString(Constants.FIRMWARE);
    this.uniqueId = bundle.getString(Constants.UNIQUE_ID);
    this.batteryPower = bundle.getInt(Constants.Devices.BATTERY);
    this.shuffled = bundle.getBoolean(SHUFFLED, false);
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  BeaconDevice(Builder builder) {
    this.address = builder.address;
    this.proximityUUID = builder.proximityUUID;
    this.major = builder.major;
    this.minor = builder.minor;
    this.txPower = builder.txPower;
    this.name = builder.name;
    this.uniqueId = builder.uniqueId;
    this.firmwareVersion = builder.firmwareVersion;
    this.batteryPower = builder.batteryPower;
    this.shuffled = builder.shuffled;
    this.rssi = builder.rssi;
    this.distance = builder.distance;
    this.proximity = builder.proximity;
    this.timestamp = builder.timestamp;
    this.password = builder.password;
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  /**
   * Creates BeaconDevice from {@link BeaconDevice} and {@link ResolvedId}
   * <p>
   * Should be used when resolving shuffled beacons manually
   * <br>
   *
   * @param beaconDevice source beacon device
   * @param resolvedId target resolved values
   * @return new beacon device
   */
  public static BeaconDevice of(IBeaconDevice beaconDevice, ResolvedId resolvedId) {
    return new BeaconDevice(beaconDevice, resolvedId);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel output, int flags) {
    final Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putInt(PARCEL_RSSI, rssi);
    bundle.putString(PARCEL_ADDRESS, address);
    bundle.putString(PARCEL_NAME, name);
    bundle.putSerializable(Constants.Devices.PROXIMITY, proximityUUID);
    bundle.putInt(Constants.Devices.MAJOR, major);
    bundle.putInt(Constants.Devices.MINOR, minor);
    bundle.putInt(Constants.Devices.TX_POWER, txPower);
    bundle.putDouble(Constants.Devices.ACCURACY, distance);
    bundle.putByteArray(Constants.Devices.PASSWORD, password);
    bundle.putLong(PARCEL_TIMESTAMP, timestamp);
    bundle.putString(Constants.FIRMWARE, firmwareVersion);
    bundle.putString(Constants.UNIQUE_ID, uniqueId);
    bundle.putInt(Constants.Devices.BATTERY, batteryPower);
    bundle.putBoolean(SHUFFLED, shuffled);
    output.writeBundle(bundle);
  }

  @Override
  public int compareTo(IBeaconDevice another) {
    SDKPreconditions.checkNotNull(another, "Comparing to null beacon device!");
    return Integer.valueOf(hashCode()).compareTo(another.hashCode());
  }

  @Override
  public int hashCode() {
    return hashCodeBuilder.append(address).build();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof BeaconDevice)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    final BeaconDevice beaconDevice = (BeaconDevice) o;

    if (this.proximityUUID != null &&
        beaconDevice.proximityUUID != null &&
        !proximityUUID.equals(beaconDevice.proximityUUID)) {
      return false;
    }

    if (this.address != null && beaconDevice.address != null && !this.address.equals(beaconDevice.address)) {
      return false;
    }

    if (this.uniqueId != null && beaconDevice.uniqueId != null && !this.uniqueId.equals(beaconDevice.uniqueId)) {
      return false;
    }

    if ((this.minor != beaconDevice.minor) || (this.major != beaconDevice.major)) {
      return false;
    }

    return true;
  }

  @Override
  public String toString() {
    return "iBeaconDevice{" +
        "address='" + address + '\'' +
        ", uniqueId='" + uniqueId + '\'' +
        ", proximityUUID=" + proximityUUID +
        ", major=" + major +
        ", minor=" + minor +
        ", rssi=" + rssi +
        ", shuffled=" + shuffled +
        '}';
  }

  /**
   * Gets timestamp - time when beacon device was discovered
   *
   * @return the timestamp
   */
  @Override
  public long getTimestamp() {
    return timestamp;
  }

  public void setTimestamp(long timestamp) {
    this.timestamp = timestamp;
  }

  /**
   * Gets Proximity UUID.
   *
   * @return the proximity UUID
   */
  public UUID getProximityUUID() {
    return proximityUUID;
  }

  /**
   * Gets one of three values: IMMEDIATE, NEAR, FAR.
   *
   * @return the proximity
   */
  @Override
  public Proximity getProximity() {
    return proximity;
  }

  public void setProximity(Proximity proximity) {
    this.proximity = proximity;
  }

  /**
   * Returns percentage battery power.
   *
   * @return the battery power
   */
  @Override
  public int getBatteryPower() {
    return batteryPower;
  }

  /**
   * Gets major value.
   *
   * @return the major
   */
  @Override
  public int getMajor() {
    return major;
  }

  /**
   * Gets beacon unique id.
   *
   * @return the beacon unique id
   */
  @Override
  public String getUniqueId() {
    return uniqueId;
  }

  /**
   * Gets minor value.
   *
   * @return the minor
   */
  @Override
  public int getMinor() {
    return minor;
  }

  /**
   * Gets measured power.
   *
   * @return the measured power
   */
  @Override
  public int getTxPower() {
    return txPower;
  }

  @Override
  public DeviceProfile getProfile() {
    return DeviceProfile.IBEACON;
  }

  /**
   * Gets distance. The distance is calculated according to algorithm
   * described in the Stack Overflow answer.
   *
   * @return the distance [m]
   * @see <a href="http://stackoverflow.com/questions/20416218/understanding-ibeacon-distancing" target="_blank">Stack Overflow answer</a>
   */
  @Override
  public double getDistance() {
    return distance;
  }

  public void setDistance(double distance) {
    this.distance = distance;
  }

  /**
   * Gets MAC address of beacon.
   *
   * @return the mac address
   */
  @Override
  public String getAddress() {
    return address;
  }

  /**
   * Gets password.
   *
   * @return the byte [ ]
   */
  @Override
  public byte[] getPassword() {
    return password;
  }

  /**
   * Sets authorization password.
   *
   * @param password the password
   */
  @Override
  public void setPassword(final byte[] password) {
    SDKPreconditions.checkNotNull(password, "Password is null");
    final int passwordLength = password.length;
    this.password = new byte[passwordLength];
    System.arraycopy(password, 0, this.password, 0, passwordLength);
  }

  /**
   * Gets rssi.
   *
   * @return the rssi
   */
  @Override
  public int getRssi() {
    return rssi;
  }

  public void setRssi(int rssi) {
    this.rssi = rssi;
  }

  /**
   * Gets beacon name.
   *
   * @return the name
   */
  @Override
  public String getName() {
    return name;
  }

  /**
   * Gets firmware version.
   *
   * @return the firmware version
   */
  @Override
  public String getFirmwareVersion() {
    return firmwareVersion;
  }

  /**
   * Checks is device shuffled
   *
   * @return true or false
   */
  @Override
  public boolean isShuffled() {
    return shuffled;
  }

  public static class Builder {

    String address;
    UUID proximityUUID;
    int major;
    int minor;
    int txPower;
    String name;
    String uniqueId;
    String firmwareVersion;
    int batteryPower;
    boolean shuffled;
    int rssi;
    double distance;
    Proximity proximity;
    long timestamp;
    byte[] password;

    public Builder() {
    }

    public Builder(IBeaconDevice device) {
      this.address = device.getAddress();
      this.proximityUUID = device.getProximityUUID();
      this.major = device.getMajor();
      this.minor = device.getMinor();
      this.txPower = device.getTxPower();
      this.name = device.getName();
      this.uniqueId = device.getUniqueId();
      this.firmwareVersion = device.getFirmwareVersion();
      this.batteryPower = device.getBatteryPower();
      this.shuffled = device.isShuffled();
      this.rssi = device.getRssi();
      this.distance = device.getDistance();
      this.proximity = device.getProximity();
      this.timestamp = device.getTimestamp();
      this.password = device.getPassword();
    }

    public Builder address(String address) {
      this.address = address;
      return this;
    }

    public Builder proximityUUID(UUID proximityUUID) {
      this.proximityUUID = proximityUUID;
      return this;
    }

    public Builder major(int major) {
      this.major = major;
      return this;
    }

    public Builder minor(int minor) {
      this.minor = minor;
      return this;
    }

    public Builder txPower(int txPower) {
      this.txPower = txPower;
      return this;
    }

    public Builder name(String name) {
      this.name = name;
      return this;
    }

    public Builder uniqueId(String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder firmwareRevision(String firmwareVersion) {
      this.firmwareVersion = firmwareVersion;
      return this;
    }

    public Builder batteryPower(int batteryPower) {
      this.batteryPower = batteryPower;
      return this;
    }

    public Builder shuffled(boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    public Builder rssi(int rssi) {
      this.rssi = rssi;
      return this;
    }

    public Builder distance(double distance) {
      this.distance = distance;
      return this;
    }

    public Builder proximity(Proximity proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder timestamp(long timestamp) {
      this.timestamp = timestamp;
      return this;
    }

    public Builder password(byte[] password) {
      SDKPreconditions.checkNotNull(password, "Password is null");
      final int passwordLength = password.length;
      this.password = new byte[passwordLength];
      System.arraycopy(password, 0, this.password, 0, passwordLength);
      return this;
    }

    public BeaconDevice build() {
      return new BeaconDevice(this);
    }
  }
}