package com.kontakt.sdk.android.ble.connection;

import android.util.Base64;

import com.kontakt.sdk.android.ble.security.exception.InvalidConfigException;
import com.kontakt.sdk.android.ble.security.parser.DataResponseParser;
import com.kontakt.sdk.android.ble.security.property.AbstractProperty;
import com.kontakt.sdk.android.ble.security.property.ByteArrayProperty;
import com.kontakt.sdk.android.ble.security.property.Int8Property;
import com.kontakt.sdk.android.ble.security.property.PropertyID;
import com.kontakt.sdk.android.ble.security.property.StringProperty;
import com.kontakt.sdk.android.ble.security.property.UInt16Property;
import com.kontakt.sdk.android.ble.security.property.UInt32Property;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Network;
import com.kontakt.sdk.android.common.model.PacketType;
import com.kontakt.sdk.android.common.model.PowerSaving;
import com.kontakt.sdk.android.common.model.PowerSavingFeature;
import com.kontakt.sdk.android.common.profile.DeviceProfile;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static com.kontakt.sdk.android.ble.security.parser.PropertyHelper.readInt16;
import static com.kontakt.sdk.android.ble.security.parser.PropertyHelper.readInt8;
import static com.kontakt.sdk.android.ble.security.parser.PropertyHelper.readString;
import static com.kontakt.sdk.android.ble.security.parser.PropertyHelper.readUInt32;
import static com.kontakt.sdk.android.ble.security.parser.PropertyHelper.readUUID;
import static com.kontakt.sdk.android.common.model.PacketType.EDDYSTONE_EID;
import static com.kontakt.sdk.android.common.model.PacketType.EDDYSTONE_ETLM;
import static com.kontakt.sdk.android.common.model.PacketType.EDDYSTONE_TLM;
import static com.kontakt.sdk.android.common.model.PacketType.EDDYSTONE_UID;
import static com.kontakt.sdk.android.common.model.PacketType.EDDYSTONE_URL;
import static com.kontakt.sdk.android.common.model.PacketType.IBEACON;
import static com.kontakt.sdk.android.common.model.PacketType.KONTAKT;
import static com.kontakt.sdk.android.common.model.PowerSavingFeature.LIGHT_SENSOR;
import static com.kontakt.sdk.android.common.model.PowerSavingFeature.MOTION_DETECTION;
import static com.kontakt.sdk.android.common.util.ConversionUtils.fromPowerLevelToDbm;
import static com.kontakt.sdk.android.common.util.ConversionUtils.toPowerLevel;
import static com.kontakt.sdk.android.common.util.EddystoneUtils.toHexString;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

class SecureConfigMapper {

  static final int IBEACON_PROFILE = 1;
  static final int EDDYSTONE_PROFILE = 2;
  static final long DEFAULT_SHUFFLE_INTERVAL = TimeUnit.HOURS.toMinutes(24);
  private static final float ADV_INTERVAL_FACTOR = 0.625F;
  private static final int DEFAULT_MOVE_SUSPEND_TIMEOUT = 6000;

  private static final int UID_BIT = 0x1;
  private static final int URL_BIT = 0x2;
  private static final int TLM_BIT = 0x4;
  private static final int IBEACON_BIT = 0x8;
  private static final int KONTAKT_PROFILE_BIT = 0x10;
  private static final int EID_BIT = 0x20;
  private static final int ETLM_BIT = 0x40;

  private static final int MOTION_SENSOR_BIT = 0x01;
  private static final int LIGHT_SENSOR_BIT = 0x02;

  private static final int SMART_BEACON_RSSI_SIZE = 8;
  private static final int CARD_BEACON_RSSI_SIZE = 2;
  private static final int PRO_BEACON_RSSI_SIZE = 7;

  private static final byte[] WPA_EAP = new byte[] { 0x01 };
  private static final byte[] WPA_PSK = new byte[] { 0x00 };

  static Config fromReadResponse(String beaconPassword, String response) {
    checkNotNull(response, "Response can't be null");
    checkNotNull(response, "Beacon password can't be null");

    byte[] decodedBase = Base64.decode(response, Base64.DEFAULT);

    DataResponseParser parser;
    try {
      parser = DataResponseParser.parse(decodedBase, beaconPassword);
      List<AbstractProperty<?>> properties = parser.getProperties();

      Config.Builder builder = new Config.Builder();
      PowerSaving.Builder powerSavingBuilder = new PowerSaving.Builder();

      for (AbstractProperty<?> property : properties) {
        readProperty(builder, powerSavingBuilder, property);
      }

      return builder.powerSaving(powerSavingBuilder.build()).build();
    } catch (InvalidConfigException e) {
      return null;
    }
  }

  static List<AbstractProperty<?>> toSecureProperties(Config config) {
    checkNotNull(config, "Config is null");
    validate(config);

    List<AbstractProperty<?>> properties = new ArrayList<>();

    if (config.getInterval() > 0) {
      properties.add(new UInt16Property(PropertyID.INTERVAL, (int) (config.getInterval() / ADV_INTERVAL_FACTOR)));
    }

    if (config.getTxPower() >= 0) {
      properties.add(new Int8Property(PropertyID.TX_POWER, fromPowerLevelToDbm(config.getTxPower())));
    }

    if (config.getMajor() > 0) {
      properties.add(new UInt16Property(PropertyID.MAJOR, config.getMajor()));
    }

    if (config.getMinor() > 0) {
      properties.add(new UInt16Property(PropertyID.MINOR, config.getMinor()));
    }

    if (config.getProximity() != null) {
      properties.add(new ByteArrayProperty(PropertyID.PROXIMITY_UUID, config.getProximity()));
    }

    String name = config.getName();
    if (name != null && !name.trim().isEmpty()) {
      properties.add(new StringProperty(PropertyID.NAME, name));
    }

    String namespace = config.getNamespace();
    if (namespace != null && !namespace.trim().isEmpty()) {
      properties.add(new ByteArrayProperty(PropertyID.NAMESPACE_ID, namespace));
    }

    String instanceId = config.getInstanceId();
    if (instanceId != null && !instanceId.trim().isEmpty()) {
      properties.add(new ByteArrayProperty(PropertyID.INSTANCE_ID, instanceId));
    }

    String url = config.getUrl();
    if (url != null && !url.trim().isEmpty()) {
      properties.add(new ByteArrayProperty(PropertyID.URL, config.getHexUrl()));
    }

    String password = config.getPassword();
    if (password != null && !password.trim().isEmpty()) {
      properties.add(new StringProperty(PropertyID.SET_PASSWORD, password));
    }

    if (config.getShuffled() != null) {
      if (config.isShuffled()) {
        properties.add(new UInt16Property(PropertyID.SHUFFLE_INTERVAL, (int) DEFAULT_SHUFFLE_INTERVAL));
      } else {
        properties.add(new UInt16Property(PropertyID.SHUFFLE_INTERVAL, 0));
      }
    }

    putProfiles(config, properties);
    putPackets(config, properties);
    putPowerSaving(config, properties);
    putGatewayNetwork(config, properties);
    putRssi(PropertyID.REF_TX_0M, config, properties);
    putRssi(PropertyID.REF_TX_1M, config, properties);

    return properties;
  }

  private static void putRssi(PropertyID propertyID, Config config, List<AbstractProperty<?>> properties) {
    checkArgument(propertyID == PropertyID.REF_TX_0M || propertyID == PropertyID.REF_TX_1M);

    List<Integer> rssi = (propertyID == PropertyID.REF_TX_0M ? config.getRssi0m() : config.getRssi1m());
    if (rssi != null && !rssi.isEmpty()) {
      properties.add(new ByteArrayProperty(propertyID, normalizeRssi(rssi)));
    }
  }

  private static void putProfiles(Config config, List<AbstractProperty<?>> properties) {
    List<DeviceProfile> profiles = config.getProfiles();
    if (profiles != null && !profiles.isEmpty()) {
      if (profiles.contains(DeviceProfile.IBEACON)) {
        properties.add(new Int8Property(PropertyID.ACTIVE_PROFILE, IBEACON_PROFILE));
      } else if (profiles.contains(DeviceProfile.EDDYSTONE)) {
        properties.add(new Int8Property(PropertyID.ACTIVE_PROFILE, EDDYSTONE_PROFILE));
      }
    }
  }

  private static void putPackets(Config config, List<AbstractProperty<?>> properties) {
    List<PacketType> packets = config.getPackets();
    if (packets != null && !packets.isEmpty()) {

      int mask = KONTAKT_PROFILE_BIT; // KONTAKT bit must be always on
      if (packets.contains(EDDYSTONE_UID)) mask |= UID_BIT;
      if (packets.contains(EDDYSTONE_URL)) mask |= URL_BIT;
      if (packets.contains(EDDYSTONE_TLM)) mask |= TLM_BIT;
      if (packets.contains(IBEACON)) mask |= IBEACON_BIT;
      if (packets.contains(EDDYSTONE_EID)) mask |= EID_BIT;
      if (packets.contains(EDDYSTONE_ETLM)) mask |= ETLM_BIT;

      properties.add(new UInt32Property(PropertyID.BEACON_PACKETS_MASK, mask));
    }
  }

  private static void putPowerSaving(Config config, List<AbstractProperty<?>> properties) {
    PowerSaving powerSaving = config.getPowerSaving();
    if (powerSaving == null) {
      return;
    }

    List<PowerSavingFeature> features = powerSaving.getFeatures();
    if (features != null) {
      int mask = 0;
      if (features.contains(PowerSavingFeature.MOTION_DETECTION)) {
        mask |= MOTION_SENSOR_BIT;
        properties.add(new UInt32Property(PropertyID.MOVE_SUSPEND_TIMEOUT, DEFAULT_MOVE_SUSPEND_TIMEOUT));
      }
      if (features.contains(LIGHT_SENSOR)) {
        mask |= LIGHT_SENSOR_BIT;
      }
      properties.add(new UInt32Property(PropertyID.POWER_SAVER_FEATURES_MASK, mask));
    }

    if (powerSaving.getLightSensorThreshold() != -1) {
      properties.add(new Int8Property(PropertyID.LIGHT_SENSOR_THRESHOLD, powerSaving.getLightSensorThreshold()));
    }
    if (powerSaving.getLightSensorHysteresis() != -1) {
      properties.add(new Int8Property(PropertyID.LIGHT_SENSOR_HIST, powerSaving.getLightSensorHysteresis()));
    }
    if (powerSaving.getLightSensorSamplingInterval() != -1) {
      properties.add(new UInt32Property(PropertyID.LIGHT_SENSOR_SAMPLING_INTERVAL, (int) powerSaving.getLightSensorSamplingInterval()));
    }
  }

  private static void putGatewayNetwork(Config config, List<AbstractProperty<?>> properties) {
    Network network = config.getGatewayNetwork();
    if (network == null) {
      return;
    }

    String gatewayWifiSsid = network.getName();
    if (gatewayWifiSsid != null && !gatewayWifiSsid.trim().isEmpty()) {
      properties.add(new StringProperty(PropertyID.WIFI_SSID, gatewayWifiSsid));
    }

    String gatewayNetworkLogin = network.getLogin();
    if (gatewayNetworkLogin != null && !gatewayNetworkLogin.trim().isEmpty()) {
      properties.add(new StringProperty(PropertyID.WIFI_USER, gatewayNetworkLogin));
    }

    String gatewayWifiPassword = network.getPassword();
    if (gatewayWifiPassword != null && !gatewayWifiPassword.trim().isEmpty()) {
      properties.add(new StringProperty(PropertyID.WIFI_PASS, gatewayWifiPassword));
    }

    String gatewayWifiApiKey = network.getApiKey();
    if (gatewayWifiApiKey != null && !gatewayWifiApiKey.trim().isEmpty()) {
      properties.add(new StringProperty(PropertyID.API_KEY, gatewayWifiApiKey));
    }

    if (network.isEnterprise()) {
      properties.add(new Int8Property(PropertyID.WIFI_PROTOCOL, WPA_EAP));
    } else if (network.isPersonal()) {
      properties.add(new Int8Property(PropertyID.WIFI_PROTOCOL, WPA_PSK));
    }

  }

  private static void readProperty(Config.Builder builder, PowerSaving.Builder powerSavingBuilder, AbstractProperty<?> property) {
    switch (property.getPropertyID()) {
      case BEACON_ID:
        builder.uniqueId(readString(property));
        break;
      case INTERVAL:
        builder.interval((int) (readInt16(property) * ADV_INTERVAL_FACTOR));
        break;
      case TX_POWER:
        builder.txPower(toPowerLevel(readInt8(property)));
        break;
      case PROXIMITY_UUID:
        builder.proximity(readUUID(property));
        break;
      case MAJOR:
        builder.major(readInt16(property));
        break;
      case MINOR:
        builder.minor(readInt16(property));
        break;
      case NAME:
        builder.name(readString(property));
        break;
      case NAMESPACE_ID:
        builder.namespace(toHexString(property.getBytes()));
        break;
      case INSTANCE_ID:
        builder.instanceId(toHexString(property.getBytes()));
        break;
      case URL:
        builder.url(toHexString(property.getBytes()));
        break;
      case SHUFFLE_INTERVAL:
        builder.shuffled(readInt16(property) > 0);
        break;
      case BEACON_PACKETS_MASK:
        readPackets(builder, readUInt32(property));
        break;
      case ACTIVE_PROFILE:
        readProfile(builder, readInt8(property));
        break;
      case POWER_SAVER_FEATURES_MASK:
        readPowerSaving(powerSavingBuilder, readUInt32(property));
        break;
      case LIGHT_SENSOR_THRESHOLD:
        powerSavingBuilder.lightSensorThreshold(readInt8(property));
        break;
      case LIGHT_SENSOR_HIST:
        powerSavingBuilder.lightSensorHysteresis(readInt8(property));
        break;
      case LIGHT_SENSOR_SAMPLING_INTERVAL:
        powerSavingBuilder.lightSensorSamplingInterval(readUInt32(property));
        break;
      case MOVE_SUSPEND_TIMEOUT:
        powerSavingBuilder.moveSuspendTimeout(readUInt32(property));
        break;
      case SET_PASSWORD:
        builder.password(readString(property));
        break;
    }
  }

  private static void readPackets(Config.Builder builder, int bytes) {
    List<PacketType> packets = new ArrayList<>();
    if ((bytes & UID_BIT) != 0) {
      packets.add(EDDYSTONE_UID);
    }
    if ((bytes & URL_BIT) != 0) {
      packets.add(EDDYSTONE_URL);
    }
    if ((bytes & TLM_BIT) != 0) {
      packets.add(EDDYSTONE_TLM);
    }
    if ((bytes & IBEACON_BIT) != 0) {
      packets.add(IBEACON);
    }
    if ((bytes & KONTAKT_PROFILE_BIT) != 0) {
      packets.add(KONTAKT);
    }
    if ((bytes & EID_BIT) != 0) {
      packets.add(EDDYSTONE_EID);
    }
    if ((bytes & ETLM_BIT) != 0) {
      packets.add(EDDYSTONE_TLM);
    }
    builder.packets(packets);
  }

  private static void readProfile(Config.Builder builder, int value) {
    if (value == IBEACON_PROFILE) {
      builder.profiles(Collections.singletonList(DeviceProfile.IBEACON));
    } else if (value == EDDYSTONE_PROFILE) {
      builder.profiles(Collections.singletonList(DeviceProfile.EDDYSTONE));
    }
  }

  private static void readPowerSaving(PowerSaving.Builder builder, int bytes) {
    List<PowerSavingFeature> features = new ArrayList<>();

    if ((bytes & MOTION_SENSOR_BIT) != 0) {
      features.add(PowerSavingFeature.MOTION_DETECTION);
    }

    if ((bytes & LIGHT_SENSOR_BIT) != 0) {
      features.add(LIGHT_SENSOR);
    }

    builder.features(features);
  }

  private static void validate(Config config) {

    List<DeviceProfile> profiles = config.getProfiles();

    if (profiles != null && profiles.size() > 1) {
      throw new IllegalArgumentException("Only single profile can be set at a time");
    }

    PowerSaving powerSaving = config.getPowerSaving();
    if (powerSaving != null) {
      if (powerSaving.getFeatures() != null && powerSaving.getFeatures().containsAll(EnumSet.of(LIGHT_SENSOR, MOTION_DETECTION))) {
        throw new IllegalArgumentException("Setting both Light Sensor and Motion Detection is not allowed");
      }
    }
    checkRssiArray(config.getRssi0m(), "Provided RSSI @ 0m array has invalid size.");
    checkRssiArray(config.getRssi1m(), "Provided RSSI @ 1m array has invalid size.");
  }

  private static void checkRssiArray(List<Integer> rssi, String message) {
    if (rssi != null && !rssi.isEmpty()) {
      if (rssi.size() != CARD_BEACON_RSSI_SIZE && rssi.size() != SMART_BEACON_RSSI_SIZE && rssi.size() != PRO_BEACON_RSSI_SIZE) {
        throw new IllegalArgumentException(message);
      }
    }
  }

  private static List<Integer> normalizeRssi(List<Integer> inputRssi) {
    //Int array passed to a device must always have 8 elements.
    List<Integer> normalized = new ArrayList<>();

    //Smart Beacon
    if (inputRssi.size() == SMART_BEACON_RSSI_SIZE) {
      normalized.addAll(inputRssi);
    }

    if (inputRssi.size() == CARD_BEACON_RSSI_SIZE) {
      for (int i = 0; i <= 7; i++) {
        if (i == 1) {
          normalized.add(inputRssi.get(0));
          continue;
        }
        if (i == 6) {
          normalized.add(inputRssi.get(1));
          continue;
        }
        normalized.add(0);
      }
    }

    if (inputRssi.size() == PRO_BEACON_RSSI_SIZE) {
      normalized.add(0);
      for (int i = 0; i <= 6; i++) {
        normalized.add(inputRssi.get(i));
      }
    }

    return normalized;
  }
}
