package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.content.Context;
import android.os.Build;
import android.os.RemoteException;
import android.util.Base64;

import com.kontakt.sdk.android.ble.device.KontaktDeviceCharacteristics;
import com.kontakt.sdk.android.ble.dfu.DfuAuthorizationService;
import com.kontakt.sdk.android.ble.dfu.DfuController;
import com.kontakt.sdk.android.ble.dfu.DfuControllerImpl;
import com.kontakt.sdk.android.ble.dfu.FirmwareUpdateListener;
import com.kontakt.sdk.android.ble.dfu.firmwares.FirmwareFilesManager;
import com.kontakt.sdk.android.ble.dfu.firmwares.IFirmwareFilesManager;
import com.kontakt.sdk.android.ble.security.auth.AuthToken;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.FirmwareRevisions;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.model.Network;
import com.kontakt.sdk.android.common.model.Preset;
import com.kontakt.sdk.android.common.model.Time;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.EddystonePropertyValidator;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.IBeaconPropertyValidator;
import com.kontakt.sdk.android.common.util.SecureProfileUtils;

import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.Set;
import java.util.UUID;

import static com.kontakt.sdk.android.ble.connection.ErrorCause.FEATURE_NOT_SUPPORTED;
import static com.kontakt.sdk.android.ble.connection.ErrorCause.GATT_FAILURE;
import static com.kontakt.sdk.android.ble.connection.ErrorCause.INCORRECT_VALUE;
import static com.kontakt.sdk.android.common.util.ConversionUtils.hexStringToByteArray;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNullOrEmpty;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;
import static java.nio.ByteOrder.LITTLE_ENDIAN;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
class KontaktDeviceConnectionImpl implements KontaktDeviceConnection, KontaktGatewayConnection {

  private static final String NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID = "00002902-0000-1000-8000-00805f9b34fb";
  private static final Set<State> CONNECTION_AVAILABLE_STATES =
      EnumSet.of(State.CONNECTED, State.AUTHENTICATED, State.AUTHENTICATING, State.CHARACTERISTICS_REQUESTING);

  volatile State state = State.DISCONNECTED;
  GattController gattController;
  private Context context;
  private RemoteBluetoothDevice device;
  private ConnectionListener connectionListener;
  private KontaktDeviceServiceStore serviceStore;
  private DfuController dfuController;
  private OfflineSecurityService offlineService;

  ReadAllListener readAllListener = ReadAllListener.noop();
  ReadListener<Time> readTimeListener = createNoopReadListener();
  ReadListener<Integer> readIntegerListener = createNoopReadListener();
  ReadListener<Network> readNetworkListener = createNoopReadListener();
  WriteListener writeListener = WriteListener.NOOP_LISTENER;
  AuthorizationCallback authorizationCallback = AuthorizationCallback.NOOP_CALLBACK;
  WriteDescriptorListener writeDescriptorListener = WriteDescriptorListener.NOOP_LISTENER;
  ChangeCharacteristicListener changeCharacteristicListener = ChangeCharacteristicListener.NOOP_LISTENER;

  KontaktDeviceConnectionImpl(final Context context, final RemoteBluetoothDevice bluetoothDevice, final ConnectionListener connectionListener) {
    KontaktSDK.getInstance();
    checkNotNull(bluetoothDevice, "Beacon device is null.");
    checkNotNull(connectionListener, "Connection listener is null.");
    validateBeaconPassword(bluetoothDevice);
    this.context = context.getApplicationContext();
    this.device = bluetoothDevice;
    this.connectionListener = connectionListener;
  }

  KontaktDeviceConnectionImpl(final Context context, final ISecureProfile kontaktSecureProfile, final ConnectionListener connectionListener) {
    this(context, SecureProfileUtils.asRemoteBluetoothDevice(kontaktSecureProfile), connectionListener);
  }

  //Used for tests
  protected KontaktDeviceConnectionImpl(final Context context, final RemoteBluetoothDevice bluetoothDevice,
                                        final ConnectionListener connectionListener, final KontaktDeviceServiceStore serviceStore) {
    this(context, bluetoothDevice, connectionListener);
    this.serviceStore = serviceStore;
  }

  //Used for tests
  protected KontaktDeviceConnectionImpl(final Context context, final ISecureProfile kontaktSecureProfile, final ConnectionListener connectionListener,
                                        final KontaktDeviceServiceStore serviceStore, final DfuController dfuController) {
    this(context, kontaktSecureProfile, connectionListener);
    this.serviceStore = serviceStore;
    this.dfuController = dfuController;
  }

  @Override
  public RemoteBluetoothDevice getDevice() {
    return device;
  }

  @Override
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public synchronized boolean connect() {
    if (!isClosed()) {
      throw new IllegalStateException("Previous connection is not closed.");
    }
    if (gattController == null) {
      try {
        gattController = GattControllerFactory.createGattController(this, context, device);
        return gattController.connect();
      } catch (RemoteException ignored) {
      }
    }
    return false;
  }

  @Override
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public synchronized boolean connect(RemoteBluetoothDevice device) {
    this.device = device;
    return connect();
  }

  @Override
  public synchronized boolean isConnected() {
    return CONNECTION_AVAILABLE_STATES.contains(state);
  }

  @Override
  public synchronized boolean isAuthenticated() {
    return state == State.AUTHENTICATED;
  }

  @Override
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public synchronized void close() {

    if (isConnected()) {
      disconnect();
    }

    Logger.v("Closing connection");

    if (dfuController != null) {
      dfuController.close();
    }
    if (gattController != null) {
      gattController.close();
    }
    if (offlineService != null) {
      offlineService.close();
    }

    unregisterAllListeners();

    gattController = null;
    dfuController = null;
    offlineService = null;
  }

  private synchronized void disconnect() {
    if (gattController != null) {
      Logger.v("Disconnecting from beacon device");
      gattController.disconnect();
    } else {
      Logger.v("Connection was not initialized");
    }
  }

  @Override
  public synchronized boolean isClosed() {
    return gattController == null;
  }

  @Override
  public synchronized void applyConfig(final Config config, final WriteBatchListener<Config> writeBatchListener) {
    checkState(!isClosed(), "Beacon connection is closed");
    checkState(isAuthenticated(), "Beacon connection is not authenticated");
    checkNotNull(writeBatchListener, "Write Batch Listener is null.");
    OperationType.CONFIG.validate(device);
    if (!isAuthenticated()) {
      writeBatchListener.onWriteFailure();
      return;
    }

    Logger.d("Applying config...");

    final RemoteBluetoothDevice.Characteristics characteristics = new KontaktDeviceCharacteristics(serviceStore);
    KontaktDeviceBatchProcessor.from(KontaktDeviceBatchProcessor.Batch.select(config, characteristics), this)
        .process(createProcessingListener(config, writeBatchListener));
  }

  @Override
  public synchronized void acceptProfile(final Preset profile, final WriteBatchListener<Preset> writeBatchListener) {
    checkState(!isClosed(), "Beacon connection is closed");
    checkState(isAuthenticated(), "Beacon connection is not authenticated");
    checkNotNull(writeBatchListener, "Write Batch Listener is null.");
    OperationType.PROFILE.validate(device);
    if (!isAuthenticated()) {
      writeBatchListener.onWriteFailure();
      return;
    }

    Logger.d("Accepting profile...");

    final RemoteBluetoothDevice.Characteristics characteristics = new KontaktDeviceCharacteristics(serviceStore);
    KontaktDeviceBatchProcessor.from(KontaktDeviceBatchProcessor.Batch.select(profile, characteristics), this)
        .process(createProcessingListener(profile, writeBatchListener));
  }

  @Override
  public synchronized void overwriteMinor(final int value, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.MINOR.validate(device);
    Logger.d(String.format("Writing new minor value %s ", String.valueOf(value)));

    try {
      IBeaconPropertyValidator.validateMinor(value);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getMinorCharacteristic(), ConversionUtils.invert(ConversionUtils.to2ByteArray(value)), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void enableNonConnectableMode(final String masterPassword, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.NON_CONNECTABLE_MODE.validate(device);
    try {
      IBeaconPropertyValidator.validateBeaconMasterPassword(masterPassword);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    Logger.d(String.format("Enabling non-connectable mode - master password: %s", masterPassword));

    try {
      overwrite(serviceStore.getNonConnectableCharacteristic(), masterPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteMajor(final int value, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.MAJOR.validate(device);
    try {
      IBeaconPropertyValidator.validateMajor(value);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
    }

    Logger.d(String.format("Writing new major value - %s", String.valueOf(value)));

    try {
      overwrite(serviceStore.getMajorCharacteristic(), ConversionUtils.invert(ConversionUtils.to2ByteArray(value)), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteProximityUUID(final UUID proximity, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.PROXIMITY_UUID.validate(device);
    try {
      checkNotNull(proximity);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getProximityCharacteristic(), ConversionUtils.convert(proximity), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwritePassword(final String newPassword, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.PASSWORD.validate(device);
    try {
      IBeaconPropertyValidator.validateBeaconPassword(newPassword, device.getFirmwareVersion());
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getSetNewPasswordCharacteristic(), newPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteModelName(final String newModelName, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.MODEL_NAME.validate(device);
    try {
      IBeaconPropertyValidator.validateModelName(newModelName);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    final byte[] modelNameBytes = newModelName.getBytes();

    Logger.d(String.format("Writing model Name - %s", newModelName));
    try {
      overwrite(serviceStore.getPropagatedDeviceNameCharacteristic(), modelNameBytes, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteAdvertisingInterval(final long millis, final WriteListener writeListener) {
    checkNotNull(writeListener, "Write Listener is null.");
    OperationType.ADVERTISING_INTERVAL.validate(device);
    try {
      IBeaconPropertyValidator.validateAdvertisingInterval((int) millis);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    Logger.d(String.format("Writing advertising interval - %s", String.valueOf(millis)));

    final int value = Double.valueOf(Math.ceil(millis / 0.625)).intValue();

    try {
      overwrite(serviceStore.getAdvertisingIntervalCharacteristic(), ConversionUtils.invert(ConversionUtils.to2ByteArray(value)), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwritePowerLevel(final int powerLevel, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.POWER_LEVEL.validate(device);
    try {
      IBeaconPropertyValidator.validatePowerLevel(powerLevel);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getPowerLevelCharacteristic(), ConversionUtils.convertPowerLevel(powerLevel), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void switchToDeviceProfile(final DeviceProfile deviceProfile, final WriteListener writeListener) {
    checkNotNull(deviceProfile, "DeviceProfile cannot be null");
    checkNotNull(writeListener, "WriteListener is null.");
    checkArgument((deviceProfile == DeviceProfile.IBEACON || deviceProfile == DeviceProfile.EDDYSTONE),
        "you can only switch to iBeacon or Eddystone");
    OperationType.DEVICE_PROFILE.validate(device);
    final byte[] deviceProfileData = new byte[]{(byte) deviceProfile.getActiveProfileValue()};

    try {
      overwrite(serviceStore.getActiveProfileCharacteristic(), deviceProfileData, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void resetDevice(final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.RESET.validate(device);
    Logger.d("Resetting Beacon device...");

    try {
      overwrite(serviceStore.getResetCharacteristic(), new byte[]{0x01}, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void enableDfuMode(final String masterPassword, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null.");
    OperationType.DFU.validate(device);
    try {
      IBeaconPropertyValidator.validateBeaconMasterPassword(masterPassword);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
    }

    try {
      overwrite(serviceStore.getBootloaderCharacteristic(), masterPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteUrl(final String newUrl, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null");
    OperationType.URL.validate(device);
    try {
      EddystonePropertyValidator.validateUrl(newUrl);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    byte[] url = EddystoneUtils.serializeUrl(newUrl);
    try {
      overwrite(serviceStore.getUrlCharacteristic(), url, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteNamespaceId(final String namespaceId, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null");
    checkNotNullOrEmpty(namespaceId, "Namespace is null or empty");
    OperationType.NAMESPACE.validate(device);
    byte[] bytes;
    try {
      EddystonePropertyValidator.validateHexString(namespaceId);
      bytes = hexStringToByteArray(namespaceId);
      EddystonePropertyValidator.validateNamespace(bytes);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getNamespaceIdCharacteristic(), bytes, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void overwriteInstanceId(final String instanceId, final WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null");
    checkNotNullOrEmpty(instanceId, "Instance id is null or empty");
    OperationType.INSTANCE_ID.validate(device);
    byte[] bytes;
    try {
      EddystonePropertyValidator.validateHexString(instanceId);
      bytes = hexStringToByteArray(instanceId);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      EddystonePropertyValidator.validateHexString(instanceId);
      EddystonePropertyValidator.validateInstanceId(bytes);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getInstanceIdCharacteristic(), bytes, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void restoreDefaultSettings(final String masterPassword, final WriteListener writeListener) {
    OperationType.RESTORE.validate(device);
    Logger.d("Restoring default settings to Beacon device...");

    try {
      IBeaconPropertyValidator.validateBeaconMasterPassword(masterPassword);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getDefaultSettingsCharacteristic(), masterPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void authorize(AuthToken token, AuthorizationCallback callback) {
    //Authorization is conducted by issuing ReadAll command.
    checkNotNull(token, "AuthToken can't be null");
    checkNotNull(callback, "AuthorizationCallback can't be null");
    OperationType.AUTHORIZATION.validate(device);
    unregisterAllListeners();

    registerAuthorizationCallback(callback);

    byte[] decoded = Base64.decode(token.getToken(), Base64.DEFAULT);

    try {
      overwriteSecure(serviceStore.getSecureWriteCharacteristic(), decoded, WriteListener.NOOP_LISTENER, true);
    } catch (Exception e) {
      authorizationCallback.onFailure(FEATURE_NOT_SUPPORTED);
      unregisterAllListeners();
    }
  }

  @Override
  public synchronized void syncTime(WriteListener listener) {
    checkNotNull(listener, "Listener is null.");
    OperationType.SYNC_TIME.validate(device);
    unregisterAllListeners();

    try {
      Time time = Time.getCurrentUTC();
      overwriteSecure(serviceStore.getDeviceTimeCharacteristic(), time.toBleValue(), listener, false);
    } catch (Exception e) {
      listener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void readAll(AuthToken token, ReadListener<Config> listener) {
    checkNotNull(token, "AuthToken is null");
    checkNotNull(listener, "Listener is null");
    OperationType.SECURE_COMMAND.validate(device);
    unregisterAllListeners();

    registerReadAllListener(ReadAllListener.create(listener, token.getPassword()));
    byte[] decoded = Base64.decode(token.getToken(), Base64.DEFAULT);

    try {
      overwriteSecure(serviceStore.getSecureWriteCharacteristic(), decoded, WriteListener.NOOP_LISTENER, true);
    } catch (Exception e) {
      readAllListener.onReadFailure(FEATURE_NOT_SUPPORTED);
      unregisterAllListeners();
    }
  }

  @Override
  public synchronized void readTime(ReadListener<Time> listener) {
    checkNotNull(listener, "Listener is null");
    OperationType.READ_TIME.validate(device);
    unregisterAllListeners();

    BluetoothDeviceCharacteristic characteristic = null;
    try {
      characteristic = serviceStore.getDeviceTimeCharacteristic();
    } catch (Exception e) {
      listener.onReadFailure(FEATURE_NOT_SUPPORTED);
      unregisterAllListeners();
    }

    registerReadTimeListener(listener);

    if (!gattController.readCharacteristic(characteristic)) {
      listener.onReadFailure(GATT_FAILURE);
      unregisterAllListeners();
    }
  }

  @Override
  public synchronized void readLightSensor(ReadListener<Integer> listener) {
    //Based on notifications to decrease light sampling interval in a beacon
    checkNotNull(listener, "Listener is null");
    OperationType.READ_LIGHT_SENSOR.validate(device);
    unregisterAllListeners();

    BluetoothDeviceCharacteristic characteristic;
    BluetoothGattDescriptor descriptor;
    try {
      characteristic = serviceStore.getLightSensorCharacteristic();
      descriptor = characteristic.getDescriptor(UUID.fromString(NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID));
      descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
    } catch (Exception e) {
      listener.onReadFailure(FEATURE_NOT_SUPPORTED);
      unregisterAllListeners();
      return;
    }

    registerReadIntListener(listener);

    if (!gattController.setCharacteristicNotification(characteristic, true) || !gattController.writeDescriptor(descriptor)) {
      listener.onReadFailure(GATT_FAILURE);
      unregisterAllListeners();
    }
  }

  @Override
  public synchronized void applySecureConfig(String encryptedConfig, WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null");
    checkNotNullOrEmpty(encryptedConfig, "Secure config is null or empty");
    unregisterAllListeners();

    OperationType.SECURE_CONFIG.validate(device);

    byte[] decodedConfig = Base64.decode(encryptedConfig, Base64.DEFAULT);

    try {
      overwriteSecure(serviceStore.getSecureWriteCharacteristic(), decodedConfig, writeListener, true);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public synchronized void applySecureConfig(Config config, AuthToken token, WriteListener writeListener) {
    checkNotNull(config, "Config is null");
    checkNotNull(token, "AuthToken is null");
    checkNotNull(writeListener, "Listener is null");
    unregisterAllListeners();

    if (offlineService == null) {
      offlineService = new OfflineSecurityService(this);
    }
    offlineService.applySecureConfig(config, token, writeListener);
  }

  @Override
  public void updateFirmware(Firmware firmware, KontaktCloud kontaktCloud, FirmwareUpdateListener firmwareUpdateListener) {
    updateFirmware(firmware, null, kontaktCloud, firmwareUpdateListener);
  }

  @Override
  public void updateFirmware(Firmware firmware, byte[] fileBytes, KontaktCloud kontaktCloud, FirmwareUpdateListener firmwareUpdateListener) {
    checkState(!isClosed(), "The connection is closed");
    checkState(isAuthenticated(), "The device is not authenticated");
    checkNotNull(kontaktCloud, "KontaktCloud is null");
    checkNotNull(firmware, "Firmware is null");
    checkNotNull(firmwareUpdateListener, "FirmwareUpdateListener is null");
    checkArgument(device.getProfile() == DeviceProfile.KONTAKT_SECURE,
        "Firmware update is meant only for Kontakt.io secure profile devices (Beacon PRO and above)");

    if (dfuController == null) {
      IFirmwareFilesManager firmwareFilesManager = FirmwareFilesManager.create(context, kontaktCloud);
      DfuAuthorizationService authorizationService = DfuAuthorizationService.create(device, kontaktCloud, gattController, serviceStore);
      dfuController = DfuControllerImpl.create(fileBytes, firmware, gattController, serviceStore, firmwareFilesManager, authorizationService);
    }
    dfuController.setFirmwareUpdateListener(firmwareUpdateListener);

    registerWriteListener(dfuController);
    registerChangeListener(dfuController);
    registerWriteDescriptorListener(dfuController);

    dfuController.initialize();
  }

  //Gateway

  @Override
  public void readNetworksCount(ReadListener<Integer> listener) {
    checkNotNull(listener, "Listener is null");
    OperationType.GATEWAY_READ_NETWORK_COUNT.validate(device);
    unregisterAllListeners();

    BluetoothDeviceCharacteristic characteristic = null;
    try {
      characteristic = serviceStore.getGatewayNetworkCountCharacteristic();
    } catch (Exception e) {
      listener.onReadFailure(FEATURE_NOT_SUPPORTED);
      unregisterAllListeners();
    }

    registerReadIntListener(listener);

    if (!gattController.readCharacteristic(characteristic)) {
      listener.onReadFailure(GATT_FAILURE);
      unregisterAllListeners();
    }
  }

  @Override
  public void selectNetworkToRead(int index, WriteListener listener) {
    checkNotNull(listener, "WriteListener is null.");
    OperationType.GATEWAY_SELECT_NETWORK.validate(device);

    try {
      byte[] value = ByteBuffer.allocate(4).order(LITTLE_ENDIAN).putInt(index).array();
      overwrite(serviceStore.getGatewayNetworkIndexCharacteristic(), value, listener);
    } catch (Exception e) {
      listener.onWriteFailure(ErrorCause.FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public void readSelectedNetwork(ReadListener<Network> listener) {
    checkNotNull(listener, "Listener is null");
    OperationType.GATEWAY_READ_NETWORK.validate(device);
    unregisterAllListeners();

    BluetoothDeviceCharacteristic characteristic = null;
    try {
      characteristic = serviceStore.getGatewayNetworkRecordCharacteristic();
    } catch (Exception e) {
      listener.onReadFailure(FEATURE_NOT_SUPPORTED);
      unregisterAllListeners();
    }

    registerReadNetworksListener(listener);

    if (!gattController.readCharacteristic(characteristic)) {
      listener.onReadFailure(GATT_FAILURE);
      unregisterAllListeners();
    }
  }

  @Override
  public synchronized void executeSecureCommand(String secureCommand, WriteListener writeListener) {
    checkNotNull(writeListener, "WriteListener is null");
    checkNotNullOrEmpty(secureCommand, "Secure command is null or empty");

    OperationType.SECURE_COMMAND.validate(device);

    byte[] decodedCommand = Base64.decode(secureCommand, Base64.DEFAULT);

    try {
      overwriteSecure(serviceStore.getSecureWriteCharacteristic(), decodedCommand, writeListener, false);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  protected synchronized void overwrite(final BluetoothGattCharacteristic characteristic, final byte[] newValue, final WriteListener listener) {
    checkState(!isClosed(), "The connection is closed");
    checkState(isAuthenticated(), "The device is not authenticated");

    registerWriteListener(listener);
    final byte[] currentValue = characteristic.getValue();
    characteristic.setValue(newValue);
    Logger.v(String.format("Writing value %s to characteristic %s", new String(newValue), characteristic.getUuid().toString()));
    if (!gattController.writeCharacteristic(characteristic)) {
      characteristic.setValue(currentValue);
      writeListener.onWriteFailure(GATT_FAILURE);
      unregisterAllListeners();
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  protected synchronized void overwriteSecure(final BluetoothGattCharacteristic characteristic, final byte[] newValue, final WriteListener listener,
                                              final boolean readResponse) {
    checkState(!isClosed(), "The connection is closed");
    checkState(isAuthenticated(), "The device is not authenticated");

    registerWriteListener(listener);
    final byte[] currentValue = characteristic.getValue();
    characteristic.setValue(newValue);
    Logger.v(String.format("Writing value %s to characteristic %s", new String(newValue), characteristic.getUuid().toString()));
    if (!gattController.writeCharacteristic(characteristic, readResponse)) {
      characteristic.setValue(currentValue);
      writeListener.onWriteFailure(GATT_FAILURE);
      readAllListener.onReadFailure(GATT_FAILURE);
      authorizationCallback.onFailure(GATT_FAILURE);
      unregisterAllListeners();
    }
  }

  State getState() {
    return state;
  }

  void finishWriteBatch() {
    unregisterAllListeners();
  }

  void onDisconnected() {
    serviceStore = null;
    connectionListener.onDisconnected();
  }

  void onDfuModeEnabled() {
    //Not implemented yet
  }

  void onServicesDiscovered(KontaktDeviceServiceStore serviceStore) {
    this.serviceStore = serviceStore;
  }

  void onAuthenticationSuccess(RemoteBluetoothDevice.Characteristics deviceCharacteristics) {
    connectionListener.onAuthenticationSuccess(deviceCharacteristics);
  }

  void onError(int errorCode) {
    connectionListener.onErrorOccured(errorCode);
  }

  void onFailure(int failureCode) {
    connectionListener.onAuthenticationFailure(failureCode);
  }

  void onConnectionStateChange(final State state) {
    this.state = state;
  }

  void onCharacteristicWritten(final boolean isSuccess, WriteListener.WriteResponse response) {
    if (isSuccess) {
      writeListener.onWriteSuccess(response);
      readAllListener.onResponseReceived(response.getExtra());
      authorizationCallback.onSuccess();
    } else {
      writeListener.onWriteFailure(GATT_FAILURE);
      readAllListener.onReadFailure(GATT_FAILURE);
      authorizationCallback.onFailure(GATT_FAILURE);
    }
  }

  void onCharacteristicRead(BluetoothDeviceCharacteristic characteristic) {
    try {
      String uuid = characteristic.getUuid().toString();
      if (KontaktDeviceCharacteristic.CURRENT_TIME.getId().equals(uuid)) {
        readTimeListener.onReadSuccess(Time.fromBleValue(characteristic.getValue()));
      } else if (KontaktDeviceCharacteristic.GATEWAY_NETWORKS_LENGTH.getId().equals(uuid)) {
        int intValue = ByteBuffer.wrap(characteristic.getValue()).order(LITTLE_ENDIAN).getInt();
        readIntegerListener.onReadSuccess(intValue);
      } else if (KontaktDeviceCharacteristic.GATEWAY_NETWORKS_RECORD.getId().equals(uuid)) {
        byte[] value = characteristic.getValue();
        readNetworkListener.onReadSuccess(Network.fromBleValue(value));
      }
    } catch (Exception e) {
      readTimeListener.onReadFailure(FEATURE_NOT_SUPPORTED);
      readIntegerListener.onReadFailure(FEATURE_NOT_SUPPORTED);
      readNetworkListener.onReadFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  void onCharacteristicChanged(BluetoothDeviceCharacteristic characteristic) {
    String uuid = characteristic.getUuid().toString();
    if (KontaktDeviceCharacteristic.LIGHT_SENSOR_PERCENTAGE.getId().equals(uuid)) {
      readIntegerListener.onReadSuccess(characteristic.getIntValue());
    } else {
      changeCharacteristicListener.onCharacteristicChanged(characteristic);
    }
  }

  void onDescriptorWritten(final boolean isSuccess, BluetoothGattDescriptor descriptor) {
    if (isSuccess) {
      writeDescriptorListener.onDescriptorWriteSuccess(descriptor);
    } else {
      writeDescriptorListener.onDescriptorWriteFailure(descriptor);
    }
  }

  void onConnectionOpened() {
    connectionListener.onConnectionOpened();
  }

  void notifyDataSetChanged() {
    connectionListener.onCharacteristicsUpdated(new KontaktDeviceCharacteristics(serviceStore));
  }

  private void registerAuthorizationCallback(final AuthorizationCallback callback) {
    this.authorizationCallback = callback;
  }

  private void registerWriteListener(final WriteListener writeListener) {
    this.writeListener = writeListener;
  }

  private void registerChangeListener(final ChangeCharacteristicListener listener) {
    changeCharacteristicListener = listener;
  }

  private void registerWriteDescriptorListener(final WriteDescriptorListener listener) {
    writeDescriptorListener = listener;
  }

  private void registerReadAllListener(final ReadAllListener readListener) {
    this.readAllListener = readListener;
  }

  private void registerReadTimeListener(final ReadListener<Time> readListener) {
    this.readTimeListener = readListener;
  }

  private void registerReadIntListener(final ReadListener<Integer> readListener) {
    this.readIntegerListener = readListener;
  }

  private void registerReadNetworksListener(final ReadListener<Network> readListener) {
    this.readNetworkListener = readListener;
  }

  void unregisterAllListeners() {
    writeListener = WriteListener.NOOP_LISTENER;
    changeCharacteristicListener = ChangeCharacteristicListener.NOOP_LISTENER;
    writeDescriptorListener = WriteDescriptorListener.NOOP_LISTENER;
    authorizationCallback = AuthorizationCallback.NOOP_CALLBACK;
    readTimeListener = createNoopReadListener();
    readAllListener = ReadAllListener.noop();
  }

  private void validateBeaconPassword(RemoteBluetoothDevice remoteBluetoothDevice) {
    if (FirmwareRevisions.NORMAL_MODE_FIRMWARE_VERSIONS.contains(remoteBluetoothDevice.getFirmwareVersion())) {
      checkNotNull(remoteBluetoothDevice.getPassword(), "Beacon password is null or empty.");
      IBeaconPropertyValidator.validateBeaconPassword(new String(remoteBluetoothDevice.getPassword()), remoteBluetoothDevice.getFirmwareVersion());
    }
  }

  private static <T> KontaktDeviceBatchProcessor.ProcessingListener createProcessingListener(final T batchHolder,
                                                                                             final WriteBatchListener<T> writeBatchListener) {
    return new KontaktDeviceBatchProcessor.ProcessingListener() {
      @Override
      public void onStart() {
        writeBatchListener.onWriteBatchStart(batchHolder);
      }

      @Override
      public void onError(int errorCode) {
        writeBatchListener.onErrorOccured(errorCode);
      }

      @Override
      public void onFinish() {
        writeBatchListener.onWriteBatchFinish(batchHolder);
      }

      @Override
      public void onRollbackStart() {

      }

      @Override
      public void onRollbackFinish() {

      }

      @Override
      public void onRollbackError(int errorCode) {

      }
    };
  }

  private static <T> ReadListener<T> createNoopReadListener() {
    return new ReadListener<T>() {
      @Override
      public void onReadSuccess(T value) {
        //No operation
      }

      @Override
      public void onReadFailure(ErrorCause cause) {
        //No operation
      }
    };
  }

  /**
   * State enum describes all possible state that BeaconConnection may be in.
   */
  public enum State {
    /**
     * Reached if the connection with Beacon was
     * established.
     */
    CONNECTED,
    /**
     * Reached if Android device sent authentication password to Beacon.
     */
    AUTHENTICATING,
    /**
     * Reached if Android device is awaiting for all characteristics that has
     * requested.
     */
    CHARACTERISTICS_REQUESTING,
    /**
     * Reached if Authentication ended successfully and all requested
     * characteristics are obtained from Beacon.
     */
    AUTHENTICATED,
    /**
     * Reached if Android device disconnected from Beacon.
     */
    DISCONNECTED
  }
}
