package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.IBeaconFutureId;
import com.kontakt.sdk.android.common.model.IBeaconId;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

class IBeaconApiResolveStrategy implements IBeaconResolveStrategy {

  private final KontaktCloud kontaktCloud;

  IBeaconApiResolveStrategy(KontaktCloud kontaktCloud) {
    this.kontaktCloud = kontaktCloud;
  }

  @Override
  public List<IBeaconFutureId> resolve(Map<IBeaconId, IBeaconResolveRequest> requests) throws Exception {
    final List<IBeaconId> toResolve = iBeaconsToResolve(requests);
    if (toResolve.isEmpty()) {
      return Collections.emptyList();
    }

    final List<IBeaconFutureId> futureIds = resolveIBeacons(toResolve);
    updateRequests(requests, futureIds);
    return futureIds;
  }

  private List<IBeaconId> iBeaconsToResolve(Map<IBeaconId, IBeaconResolveRequest> requests) {
    final List<IBeaconId> toResolve = new ArrayList<>();
    for (Map.Entry<IBeaconId, IBeaconResolveRequest> entry : requests.entrySet()) {
      IBeaconResolveRequest request = entry.getValue();
      if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
        continue;
      }
      toResolve.add(entry.getKey());
    }
    return toResolve;
  }

  private List<IBeaconFutureId> resolveIBeacons(final List<IBeaconId> ids) throws IOException, KontaktCloudException {
    final Devices cloudResult = kontaktCloud.devices().iBeacons().withIds(ids).execute();
    final List<Device> devices = cloudResult.getContent();

    final List<IBeaconFutureId> futureIds = new ArrayList<>();
    for (Device device : devices) {
      IBeaconFutureId iBeaconFutureId = device.extractIBeaconFutureId();
      if (iBeaconFutureId != null) {
        futureIds.add(iBeaconFutureId);
      }
    }
    return futureIds;
  }

  private void updateRequests(Map<IBeaconId, IBeaconResolveRequest> requests, List<IBeaconFutureId> futureIds) {
    for (IBeaconFutureId id : futureIds) {
      updateRequest(requests, id);
    }
  }

  private void updateRequest(Map<IBeaconId, IBeaconResolveRequest> requests, IBeaconFutureId id) {
    IBeaconResolveRequest request = requests.get(id.getQueriedBy());
    request.setResolvedBy(ResolverType.API);
    if (ResolveRequestStatus.IGNORED == request.getStatus()) {
      return;
    }
    request.setStatus(ResolveRequestStatus.RESOLVED);
  }
}
