package com.kontakt.sdk.android.cloud.api.executor.venues;

import android.util.Base64;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.VenuesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Venue;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link VenuesApi}. Use this class if you want to create
 * venues through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Venue venue = new Venue.Builder()
 *      .name("Kontakt.io office")
 *      .description("Beacon valley")
 *      .build();
 *   kontaktCloud.venues().create(venue)
 *      .withImageFile(new File(fileName))
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that new venue instance that we want to create must have initialized name and
 * description fields. Otherwise an exception will be thrown.
 */
public class CreateVenueRequestExecutor extends RequestExecutor<Venue> {

  private final VenuesService venuesService;
  private final Venue venue;
  private String imageFile;

  /**
   * Constructs request executor initialized with corresponding service class and venue object.
   *
   * @param venuesService the venues API service.
   * @param venue the venue entity.
   */
  public CreateVenueRequestExecutor(final VenuesService venuesService, final Venue venue) {
    this.venuesService = venuesService;
    this.venue = venue;
  }

  /**
   * Specifies an image file of newly created venue.
   *
   * @param file the image file.
   * @return this request executor.
   * @throws IOException instance of exception
   */
  public CreateVenueRequestExecutor withImageFile(final File file) throws IOException {
    SDKPreconditions.checkNotNull(file, "file is null");
    SDKPreconditions.checkState(file.exists(), "file does not exist");
    final byte[] bytes = ConversionUtils.convert(file);
    this.imageFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * Specifies an image file of newly created venue.
   *
   * @param bytes the image file as a bytes stream.
   * @return this request executor.
   * @throws IOException instance of IOException
   */
  public CreateVenueRequestExecutor withImageFile(final byte[] bytes) throws IOException {
    SDKPreconditions.checkNotNull(bytes, "bytes cannot be null");
    this.imageFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Venue execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Venue> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Venue> prepareCall() {
    return venuesService.createVenue(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(venue.getName() != null, "cannot create venue - specify name");
    SDKPreconditions.checkState(venue.getDescription() != null, "cannot create venue - specify description");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Venues.NAME_PARAMETER, venue.getName());
    params.put(CloudConstants.Venues.DESCRIPTION_PARAMETER, venue.getDescription());
    if (imageFile != null) {
      params.put(CloudConstants.Venues.FILE_PARAMETER, imageFile);
    }
    if (venue.getLatitude() != null) {
      params.put(CloudConstants.Common.LAT_PARAMETER, venue.getLatitude());
    }
    if (venue.getLongitude() != null) {
      params.put(CloudConstants.Common.LNG_PARAMETER, venue.getLongitude());
    }

    return params;
  }
}
