package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothProfile;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.RemoteException;
import com.kontakt.sdk.android.ble.device.KontaktDeviceCharacteristics;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceService;
import com.kontakt.sdk.android.ble.util.BluetoothUtils;
import com.kontakt.sdk.android.common.TimestampUtil;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.io.Closeable;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.TimeUnit;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
final class NormalGattController extends BluetoothGattCallback implements Closeable, GattController {

  KontaktDeviceConnectionImpl connection;
  private final byte[] beaconPassword;
  private Context context;
  private Handler uiHandler;
  private BluetoothDevice bluetoothDevice;
  private BluetoothGatt gattServer;
  private KontaktDeviceServiceStore deviceServiceStore;
  private Queue<BluetoothDeviceCharacteristic> preAuthCharacteristicsQueue;

  NormalGattController(KontaktDeviceConnectionImpl connection, Context context, RemoteBluetoothDevice device) throws RemoteException {
    this.connection = connection;
    this.context = context;
    this.beaconPassword = device.getPassword();
    this.uiHandler = new Handler(Looper.getMainLooper());

    final BluetoothDevice bluetoothDevice = BluetoothUtils.getBluetoothDevice(device.getAddress());
    if (bluetoothDevice == null) {
      throw new RemoteException("Bluetooth device is null");
    }

    this.bluetoothDevice = bluetoothDevice;
  }

  @Override
  public boolean connect() {
    gattServer = bluetoothDevice.connectGatt(context, false, this);
    return gattServer != null;
  }

  @Override
  public void onConnectionStateChange(BluetoothGatt gatt, int status, int newState) {
    switch (newState) {
      case BluetoothProfile.STATE_CONNECTED:
        if (BluetoothGatt.GATT_SUCCESS == status) {
          connection.onConnectionStateChange(KontaktDeviceConnectionImpl.State.CONNECTED);
          connection.onConnectionOpened();
          if (!gattServer.discoverServices()) {
            connection.onError(DeviceConnectionError.ERROR_SERVICES_DISCOVERY);
          }
        } else {
          connection.onError(DeviceConnectionError.toGattError(status));
        }
        break;

      case BluetoothProfile.STATE_DISCONNECTED:
        if (connection != null) {
          if (BluetoothGatt.GATT_SUCCESS != status) {
            connection.onError(DeviceConnectionError.toGattError(status));
          }
          if (connection != null) {
            if (connection.getState() == KontaktDeviceConnectionImpl.State.AUTHENTICATING) {
              connection.onFailure(DeviceConnectionError.FAILURE_WRONG_PASSWORD);
            }
          }
          if (connection != null) {
            connection.onConnectionStateChange(KontaktDeviceConnectionImpl.State.DISCONNECTED);
          }
          if (connection != null) {
            connection.onDisconnected();
          }
        }
        break;

      default:
        throw new IllegalArgumentException("Unsupported connection state change code: " + newState);
    }
  }

  @Override
  public void onServicesDiscovered(BluetoothGatt gatt, int status) {
    Logger.d(String.format("On Services Discovered: %s ", gatt.toString()));

    if (status != BluetoothGatt.GATT_SUCCESS) {
      Logger.v("Services discovered but with no success: " + status);
      return;
    }

    try {
      deviceServiceStore = new KontaktDeviceServiceStore(gatt.getServices(), KontaktDeviceServiceStore.StoreType.NORMAL);
      connection.onServicesDiscovered(deviceServiceStore);
      if (deviceServiceStore.contains(KontaktDeviceService.DFU_SERVICE)) {
        connection.onDfuModeEnabled();
      } else {
        authorize(gatt);
      }
    } catch (RemoteException e) {
      connection.onFailure(DeviceConnectionError.FAILURE_UNKNOWN_BEACON);
    }
  }

  @Override
  public void onCharacteristicRead(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic, int status) {
    if (status == BluetoothGatt.GATT_SUCCESS) {
      final BluetoothDeviceCharacteristic wrapper = new BluetoothDeviceCharacteristic(characteristic);
      try {
        deviceServiceStore.replace(wrapper);
      } catch (ServiceAbsentException ignored) {
      }
      if (connection.getState() == KontaktDeviceConnectionImpl.State.CHARACTERISTICS_REQUESTING) {
        requestOrSetAuthenticated();
      }
    } else if (status == BluetoothGatt.GATT_READ_NOT_PERMITTED) {
      if (connection.getState() == KontaktDeviceConnectionImpl.State.CHARACTERISTICS_REQUESTING) {
        requestOrSetAuthenticated();
      }
    }
  }

  @Override
  public void onCharacteristicWrite(final BluetoothGatt gatt, final BluetoothGattCharacteristic characteristic, final int status) {
    if (connection.isConnected()) {
      try {
        deviceServiceStore.replace(new BluetoothDeviceCharacteristic(characteristic));
      } catch (ServiceAbsentException ignored) {
      }

      final boolean isSuccess = (status == BluetoothGatt.GATT_SUCCESS);
      KontaktDeviceCharacteristic beaconCharacteristic = KontaktDeviceCharacteristic.fromUuid(characteristic.getUuid());
      if (beaconCharacteristic == null) {
        return;
      }

      if (beaconCharacteristic == KontaktDeviceCharacteristic.PASSWORD) {
        if (isSuccess) {
          uiHandler.postDelayed(characteristicRequestRunnable, TimeUnit.SECONDS.toMillis(1));
          connection.onConnectionStateChange(KontaktDeviceConnectionImpl.State.AUTHENTICATING);
        } else {
          connection.onConnectionStateChange(KontaktDeviceConnectionImpl.State.CONNECTED);
          connection.onError(DeviceConnectionError.ERROR_AUTHENTICATION);
        }
      } else {
        connection.onCharacteristicWritten(isSuccess, new WriteListener.WriteResponse(TimestampUtil.getTimestamp(), null));
        if (isSuccess) {
          if (connection != null) {
            connection.notifyDataSetChanged();
          }
        }
      }
    }
  }

  @Override
  public void disconnect() {
    if (gattServer != null) {
      refresh();
      gattServer.disconnect();
    }
  }

  @Override
  public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic) {
    return gattServer.writeCharacteristic(characteristic);
  }

  @Override
  public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic, boolean readResponse) {
    throw new UnsupportedOperationException("Not supported in normal mode");
  }

  @Override
  public boolean readCharacteristic(BluetoothGattCharacteristic characteristic) {
    throw new UnsupportedOperationException("Not supported in normal mode");
  }

  @Override
  public boolean writeDescriptor(BluetoothGattDescriptor descriptor) {
    //Not used in Normal Gatt Controller
    return false;
  }

  @Override
  public boolean setCharacteristicNotification(BluetoothGattCharacteristic characteristic, boolean enable) {
    //Not used in Normal Gatt Controller
    return false;
  }

  @Override
  public void refresh() {
    BluetoothUtils.refreshGattServer(gattServer);
  }

  @Override
  public void close() {
    if (gattServer != null) {
      disconnect();
      gattServer.close();
    }

    if (deviceServiceStore != null) {
      deviceServiceStore.clear();
    }

    if (preAuthCharacteristicsQueue != null) {
      preAuthCharacteristicsQueue.clear();
    }

    connection = null;
    uiHandler.removeCallbacks(characteristicRequestRunnable);
  }

  private Runnable characteristicRequestRunnable = new Runnable() {
    @Override
    public void run() {
      if (connection == null) {
        return;
      }
      if (connection.getState() == KontaktDeviceConnectionImpl.State.AUTHENTICATING) {
        connection.onConnectionStateChange(KontaktDeviceConnectionImpl.State.CHARACTERISTICS_REQUESTING);
        requestCharacteristics();
      }
    }
  };

  private void requestOrSetAuthenticated() {
    if (preAuthCharacteristicsQueue.isEmpty()) {
      connection.onConnectionStateChange(KontaktDeviceConnectionImpl.State.AUTHENTICATED);
      connection.onAuthenticationSuccess(new KontaktDeviceCharacteristics(deviceServiceStore));
    } else {
      request(preAuthCharacteristicsQueue.poll());
    }
  }

  void requestCharacteristics() {

    Collection<BluetoothDeviceCharacteristic> readableCharacteristics = deviceServiceStore.getReadableCharacteristics();

    if (preAuthCharacteristicsQueue == null) {
      preAuthCharacteristicsQueue = new ArrayDeque<>(readableCharacteristics.size());
    } else {
      preAuthCharacteristicsQueue.clear();
    }

    preAuthCharacteristicsQueue.addAll(readableCharacteristics);

    request(preAuthCharacteristicsQueue.poll());
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void request(final BluetoothDeviceCharacteristic wrapper) {
    if (!gattServer.readCharacteristic(wrapper)) {
      if (connection != null && connection.getState() == KontaktDeviceConnectionImpl.State.CHARACTERISTICS_REQUESTING) {
        requestOrSetAuthenticated();
      }
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void authorize(final BluetoothGatt gatt) {
    try {
      final BluetoothGattCharacteristic passwordCharacteristic = deviceServiceStore.getPasswordCharacteristic();
      passwordCharacteristic.setValue(beaconPassword);

      if (!gatt.writeCharacteristic(passwordCharacteristic)) {
        connection.onError(DeviceConnectionError.ERROR_AUTHENTICATION);
      }
    } catch (Exception ignored) {
      connection.onError(DeviceConnectionError.ERROR_AUTHENTICATION);
    }
  }
}
