package com.kontakt.sdk.android.common.util;

import android.os.Parcel;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Kontakt.io Secure Profile utilities.
 */
public class SecureProfileUtils {

  /**
   * Returns Secure Profile model as RemoteBluetoothDevice and more precisely as IEddystoneDevice because secure profile contains some of the
   * Eddystone properties.
   *
   * @param secureProfile secure profile model.
   * @return {@link RemoteBluetoothDevice} instance.
   */
  public static RemoteBluetoothDevice asRemoteBluetoothDevice(final ISecureProfile secureProfile) {
    return new IEddystoneDevice() {
      @Override
      public String getNamespace() {
        return secureProfile.getNamespace();
      }

      @Override
      public String getInstanceId() {
        return secureProfile.getInstanceId();
      }

      @Override
      public String getUrl() {
        return null;
      }

      @Override
      public String getEid() {
        return null;
      }

      @Override
      public Telemetry getTelemetry() {
        return null;
      }

      @Override
      public String getEncryptedTelemetry() {
        return null;
      }

      @Override
      public double getDistance() {
        return 0;
      }

      @Override
      public long getTimestamp() {
        return System.currentTimeMillis();
      }

      @Override
      public Proximity getProximity() {
        return Proximity.UNKNOWN;
      }

      @Override
      public int getRssi() {
        return secureProfile.getRssi();
      }

      @Override
      public String getFirmwareVersion() {
        return secureProfile.getFirmwareRevision();
      }

      @Override
      public String getName() {
        return secureProfile.getName();
      }

      @Override
      public String getUniqueId() {
        return secureProfile.getUniqueId();
      }

      @Override
      public int getBatteryPower() {
        return secureProfile.getBatteryLevel();
      }

      @Override
      public int getTxPower() {
        return secureProfile.getTxPower();
      }

      @Override
      public DeviceProfile getProfile() {
        return DeviceProfile.KONTAKT_SECURE;
      }

      @Override
      public boolean isShuffled() {
        return secureProfile.isShuffled();
      }

      @Override
      public String getAddress() {
        return secureProfile.getMacAddress();
      }

      @Override
      public byte[] getPassword() {
        return new byte[0];
      }

      @Override
      public void setPassword(byte[] password) {
        //Ignore
      }

      @Override
      public int describeContents() {
        return 0;
      }

      @Override
      public int compareTo(IEddystoneDevice o) {
        return 0;
      }

      @Override
      public void writeToParcel(Parcel parcel, int i) {
        //Ignore
      }
    };
  }

  /**
   * Returns Secure Profile models list as RemoteBluetoothDevice list.
   *
   * @param iSecureProfiles secure profile models list.
   * @return List of Secure Profiles wrapped as {@link RemoteBluetoothDevice}.
   */
  public static List<RemoteBluetoothDevice> asRemoteBluetoothDevices(final List<ISecureProfile> iSecureProfiles) {
    checkNotNull(iSecureProfiles, "List is null.");
    if (iSecureProfiles.isEmpty()) {
      return Collections.emptyList();
    }
    List<RemoteBluetoothDevice> devices = new ArrayList<>();
    for (ISecureProfile secureProfile : iSecureProfiles) {
      devices.add(asRemoteBluetoothDevice(secureProfile));
    }
    return devices;
  }

  /**
   * Returns SecureProfile model created from RemoteBluetoothDevices
   *
   * @param device remote bluetooth device.
   * @return ISecureProfile device.
   */
  public static ISecureProfile fromRemoteBluetoothDevice(RemoteBluetoothDevice device) {
    checkNotNull(device, "Device is null.");
    return new SecureProfile.Builder(device).build();
  }

  /**
   * Returns list of SecureProfile models created from RemoteBluetoothDevices
   *
   * @param devices remote bluetooth devices list.
   * @return List of SecureProfile models.
   */
  public static List<ISecureProfile> fromRemoteBluetoothDevices(final List<? extends RemoteBluetoothDevice> devices) {
    checkNotNull(devices, "List is null.");
    if (devices.isEmpty()) {
      return Collections.emptyList();
    }
    List<ISecureProfile> secureProfiles = new ArrayList<>();
    for (RemoteBluetoothDevice device : devices) {
      secureProfiles.add(fromRemoteBluetoothDevice(device));
    }
    return secureProfiles;
  }
}
