package com.kontakt.sdk.android.common.util;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import java.util.Arrays;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Simple helper class that checks if given permission is granted. Throws an exception if otherwise.
 */
public final class PermissionsUtils {

  private PermissionsUtils() {
    throw new AssertionError();
  }

  public static void checkPermissions(Context context, String[] permissions) {
    checkNotNull(context, "Context cannot be null");
    checkNotNull(permissions, "Permissions array cannot be null");
    for (final String permission : permissions) {
      if (context.checkCallingOrSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
        throw new RuntimeException(String.format("Permission %s is not granted", permission));
      }
    }
  }

  public static void checkAtLeastOnePermissionGranted(Context context, String[] permissions) {
    checkNotNull(context, "Context cannot be null");
    checkNotNull(permissions, "Permissions array cannot be null");

    for (String permission : permissions) {
      if (context.checkCallingOrSelfPermission(permission) == PackageManager.PERMISSION_GRANTED) {
        return;
      }
    }
    throw new RuntimeException(String.format("At least one of %s must be granted for Android Marshmallow or higher", Arrays.toString(permissions)));
  }

  public static void checkAtLeastOneMarshmallowPermissionGranted(Context context, String[] permissions) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
      checkAtLeastOnePermissionGranted(context, permissions);
    }
  }
}
