package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Model class that represents JSON structure of Kontakt.io trigger's context.
 * To create new instance of this class, please use {@link Trigger.Builder}.
 */
public class TriggerContext implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<TriggerContext> CREATOR = new Creator<TriggerContext>() {
    @Override
    public TriggerContext createFromParcel(Parcel source) {
      return new TriggerContext(source);
    }

    @Override
    public TriggerContext[] newArray(int size) {
      return new TriggerContext[size];
    }
  };

  private final String cooldown;
  private final String sourceId;
  private final Proximity proximity;
  private final String trackingId;
  private final UUID proximityUuid;
  private final int major;
  private final int minor;
  private final String namespace;
  private final String instanceId;

  public static Builder builder() {
    return new Builder();
  }

  TriggerContext(Builder builder) {
    this.cooldown = builder.cooldown;
    this.sourceId = builder.sourceId;
    this.proximity = builder.proximity;
    this.trackingId = builder.trackingId;
    this.proximityUuid = builder.proximityUUID;
    this.major = builder.major;
    this.minor = builder.minor;
    this.namespace = builder.namespace;
    this.instanceId = builder.instanceId;
  }

  protected TriggerContext(Parcel in) {
    this.cooldown = in.readString();
    this.sourceId = in.readString();
    int tmpProximity = in.readInt();
    this.proximity = tmpProximity == -1 ? null : Proximity.values()[tmpProximity];
    this.trackingId = in.readString();
    this.proximityUuid = (UUID) in.readSerializable();
    this.major = in.readInt();
    this.minor = in.readInt();
    this.namespace = in.readString();
    this.instanceId = in.readString();
  }

  private TriggerContext() {
    this(new Builder());
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof TriggerContext)) {
      return false;
    }
    final TriggerContext triggerContext = (TriggerContext) object;

    return SDKEqualsBuilder.start()
        .equals(cooldown, triggerContext.cooldown)
        .equals(sourceId, triggerContext.sourceId)
        .equals(proximity, triggerContext.proximity)
        .equals(trackingId, triggerContext.trackingId)
        .equals(proximityUuid, triggerContext.proximityUuid)
        .equals(major, triggerContext.major)
        .equals(minor, triggerContext.minor)
        .equals(namespace, triggerContext.namespace)
        .equals(instanceId, triggerContext.instanceId)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(cooldown)
        .append(sourceId)
        .append(proximity)
        .append(trackingId)
        .append(proximityUuid)
        .append(major)
        .append(minor)
        .append(namespace)
        .append(instanceId)
        .build();
  }

  @Override
  public String toString() {
    List<String> components = new ArrayList<>();
    if (!TextUtils.isEmpty(trackingId)) {
      components.add("trackingId: " + trackingId);
    }
    if (proximityUuid != null) {
      components.add("proximity UUID: " + proximityUuid.toString());
    }
    if (major != BeaconRegion.ANY_MAJOR) {
      components.add("major: " + major);
    }
    if (minor != BeaconRegion.ANY_MINOR) {
      components.add("minor: " + minor);
    }
    if (!TextUtils.isEmpty(namespace)) {
      components.add("namespace: " + namespace);
    }
    if (!TextUtils.isEmpty(instanceId) && !EddystoneNamespace.ANY_INSTANCE_ID.equals(instanceId)) {
      components.add("instanceId: " + instanceId);
    }
    return "TriggerContext[" + StringUtils.join(components, ",") + "]";
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.cooldown);
    dest.writeString(this.sourceId);
    dest.writeInt(this.proximity == null ? -1 : this.proximity.ordinal());
    dest.writeString(this.trackingId);
    dest.writeSerializable(this.proximityUuid);
    dest.writeInt(this.major);
    dest.writeInt(this.minor);
    dest.writeString(this.namespace);
    dest.writeString(this.instanceId);
  }

  public String getCooldown() {
    return cooldown;
  }

  public String getSourceId() {
    return sourceId;
  }

  public Proximity getProximity() {
    return proximity;
  }

  public String getTrackingId() {
    return trackingId;
  }

  public UUID getProximityUUID() {
    return proximityUuid;
  }

  public int getMajor() {
    return major;
  }

  public int getMinor() {
    return minor;
  }

  public String getNamespace() {
    return namespace;
  }

  public String getInstanceId() {
    return instanceId;
  }

  /**
   * Builder class that is used to build {@link TriggerContext} instances from values configured by the setters.
   */
  public static class Builder {

    String cooldown;
    String sourceId;
    Proximity proximity;
    String trackingId;
    UUID proximityUUID;
    int major = BeaconRegion.ANY_MAJOR;
    int minor = BeaconRegion.ANY_MINOR;
    String namespace;
    String instanceId = EddystoneNamespace.ANY_INSTANCE_ID;

    public Builder cooldown(final String cooldown) {
      this.cooldown = cooldown;
      return this;
    }

    public Builder sourceId(final String sourceId) {
      this.sourceId = sourceId;
      return this;
    }

    public Builder proximity(final Proximity proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder trackingId(final String trackingId) {
      this.trackingId = trackingId;
      return this;
    }

    public Builder proximityUUID(final UUID proximityUUID) {
      this.proximityUUID = proximityUUID;
      return this;
    }

    public Builder major(final int major) {
      this.major = major;
      return this;
    }

    public Builder minor(final int minor) {
      this.minor = minor;
      return this;
    }

    public Builder namespace(final String namespace) {
      this.namespace = namespace;
      return this;
    }

    public Builder instanceId(final String instanceId) {
      this.instanceId = instanceId;
      return this;
    }

    public TriggerContext build() {
      return new TriggerContext(this);
    }
  }
}
