package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.List;
import java.util.UUID;

/**
 * Model class that provides information concerning the device embedded software that is currently
 * installed on it.
 * <br><br>
 * We can obtain firmwares via {@link KontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link Firmware.Builder}.
 */
public class Firmware implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Firmware> CREATOR = new Creator<Firmware>() {
    @Override
    public Firmware createFromParcel(Parcel source) {
      return new Firmware(source);
    }

    @Override
    public Firmware[] newArray(int size) {
      return new Firmware[size];
    }
  };

  private UUID id;
  private String name;
  private String description;
  private String validVersions;
  private String fileExtension;
  private boolean important;
  private DeviceType deviceType;
  private boolean scheduled;
  private boolean optional;

  @SerializedName("uniqueId") private List<String> deviceUniqueIds;

  @SerializedName("url") private String fileUrl;

  public static Builder builder() {
    return new Builder();
  }

  private Firmware() {
    this(new Builder());
  }

  Firmware(Builder builder) {
    this.id = builder.id;
    this.name = builder.name;
    this.description = builder.description;
    this.validVersions = builder.validVersions;
    this.important = builder.important;
    this.deviceType = builder.deviceType;
    this.scheduled = builder.scheduled;
    this.optional = builder.optional;
    this.deviceUniqueIds = builder.deviceUniqueIds;
    this.fileUrl = builder.fileUrl;
    this.fileExtension = builder.fileExtension;
  }

  protected Firmware(Parcel in) {
    this.id = (UUID) in.readSerializable();
    this.name = in.readString();
    this.description = in.readString();
    this.validVersions = in.readString();
    this.important = in.readByte() != 0;
    int tmpDeviceType = in.readInt();
    this.deviceType = tmpDeviceType == -1 ? null : DeviceType.values()[tmpDeviceType];
    this.scheduled = in.readByte() != 0;
    this.optional = in.readByte() != 0;
    this.deviceUniqueIds = in.createStringArrayList();
    this.fileUrl = in.readString();
    this.fileExtension = in.readString();
  }

  @Override
  public boolean equals(Object o) {
    if (o == this) {
      return true;
    }
    if (o == null || !(o instanceof Firmware)) {
      return false;
    }

    final Firmware firmware = (Firmware) o;
    return SDKEqualsBuilder.start().equals(id, firmware.id).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(id).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.id);
    dest.writeString(this.name);
    dest.writeString(this.description);
    dest.writeString(this.validVersions);
    dest.writeByte(important ? (byte) 1 : (byte) 0);
    dest.writeInt(this.deviceType == null ? -1 : this.deviceType.ordinal());
    dest.writeByte(scheduled ? (byte) 1 : (byte) 0);
    dest.writeByte(optional ? (byte) 1 : (byte) 0);
    dest.writeStringList(this.deviceUniqueIds);
    dest.writeString(this.fileUrl);
    dest.writeString(this.fileExtension);
  }

  public UUID getId() {
    return id;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public String getValidVersions() {
    return validVersions;
  }

  public boolean isImportant() {
    return important;
  }

  public DeviceType getDeviceType() {
    return deviceType;
  }

  public boolean isScheduled() {
    return scheduled;
  }

  public boolean isOptional() {
    return optional;
  }

  public List<String> getDeviceUniqueIds() {
    return deviceUniqueIds;
  }

  public String getFileUrl() {
    return fileUrl;
  }

  public String getFileExtension() {
    return fileExtension;
  }

  public static class Builder {
    UUID id;
    String name;
    String description;
    String validVersions;
    boolean important;
    DeviceType deviceType;
    boolean scheduled;
    boolean optional;
    List<String> deviceUniqueIds;
    String fileUrl;
    String fileExtension;

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder description(final String description) {
      this.description = description;
      return this;
    }

    public Builder validVersions(final String validVersions) {
      this.validVersions = validVersions;
      return this;
    }

    public Builder important(final boolean important) {
      this.important = important;
      return this;
    }

    public Builder deviceType(final DeviceType deviceType) {
      this.deviceType = deviceType;
      return this;
    }

    public Builder scheduled(final boolean scheduled) {
      this.scheduled = scheduled;
      return this;
    }

    public Builder optional(final boolean optional) {
      this.optional = optional;
      return this;
    }

    public Builder devices(final List<String> deviceUniqueIds) {
      this.deviceUniqueIds = deviceUniqueIds;
      return this;
    }

    public Builder fileUrl(final String fileUrl) {
      this.fileUrl = fileUrl;
      return this;
    }

    public Builder fileExtension(final String fileExtension) {
      this.fileExtension = fileExtension;
      return this;
    }

    public Firmware build() {
      return new Firmware(this);
    }
  }
}
