package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * A container for different events that can be collected.
 * <br><br>
 * We can collect events via {@link KontaktCloud}.
 */
public class Event implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Parcelable.Creator<Event> CREATOR = new Parcelable.Creator<Event>() {
    @Override
    public Event createFromParcel(Parcel source) {
      return new Event(source);
    }

    @Override
    public Event[] newArray(int size) {
      return new Event[size];
    }
  };

  private final EventType type;
  private final long timestamp;
  private final String uniqueId;
  private final String deviceAddress;
  private final int batteryLevel;
  private final double rssi;

  public static Builder builder() {
    return new Builder();
  }

  Event(Builder builder) {
    this.type = builder.type;
    this.timestamp = builder.timestamp / 1000;
    this.uniqueId = builder.uniqueId;
    this.deviceAddress = builder.deviceAddress;
    this.batteryLevel = builder.batteryLevel;
    this.rssi = builder.rssi;
  }

  protected Event(Parcel in) {
    int tmpEventType = in.readInt();
    this.type = tmpEventType == -1 ? null : EventType.values()[tmpEventType];
    this.timestamp = in.readLong();
    this.uniqueId = in.readString();
    this.deviceAddress = in.readString();
    this.batteryLevel = in.readInt();
    this.rssi = in.readDouble();
  }

  /**
   * Creates concrete event object based on device and event type.
   *
   * @param device the bluetooth device.
   * @param eventType the event type.
   * @return the event instance.
   */
  public static Event of(final RemoteBluetoothDevice device, final EventType eventType) {
    SDKPreconditions.checkNotNull(device, "device cannot be null");
    SDKPreconditions.checkNotNull(eventType, "event type cannot be null");

    switch (eventType) {
      case SCAN:
        return scanEvent(device);
      case MONITORING:
        return monitoringEvent(device);
      default:
        throw new IllegalArgumentException("invalid event type");
    }
  }

  private static Event scanEvent(final RemoteBluetoothDevice device) {
    return new Builder().type(EventType.SCAN)
        .timestamp(device.getTimestamp())
        .uniqueId(device.getUniqueId())
        .deviceAddress(device.getAddress())
        .rssi(device.getRssi())
        .batteryLevel(device.getBatteryPower())
        .build();
  }

  private static Event monitoringEvent(final RemoteBluetoothDevice device) {
    return new Builder().type(EventType.MONITORING)
        .timestamp(device.getTimestamp())
        .uniqueId(device.getUniqueId())
        .deviceAddress(device.getAddress())
        .batteryLevel(device.getBatteryPower())
        .build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.type == null ? -1 : this.type.ordinal());
    dest.writeLong(this.timestamp);
    dest.writeString(this.uniqueId);
    dest.writeString(this.deviceAddress);
    dest.writeInt(this.batteryLevel);
    dest.writeDouble(this.rssi);
  }

  public EventType getType() {
    return type;
  }

  public long getTimestamp() {
    return timestamp;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public String getDeviceAddress() {
    return deviceAddress;
  }

  public int getBatteryLevel() {
    return batteryLevel;
  }

  public double getRssi() {
    return rssi;
  }

  /**
   * Builder class that is used to build {@link Event} instances from values configured by the setters.
   */
  public static class Builder {
    EventType type;
    long timestamp;
    String uniqueId;
    String deviceAddress;
    int batteryLevel;
    double rssi;

    public Builder type(final EventType type) {
      this.type = type;
      return this;
    }

    public Builder timestamp(final long timestamp) {
      this.timestamp = timestamp;
      return this;
    }

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder deviceAddress(final String deviceAddress) {
      this.deviceAddress = deviceAddress;
      return this;
    }

    public Builder batteryLevel(final int batteryLevel) {
      this.batteryLevel = batteryLevel;
      return this;
    }

    public Builder rssi(final double rssi) {
      this.rssi = rssi;
      return this;
    }

    public Event build() {
      return new Event(this);
    }
  }
}
