package com.kontakt.sdk.android.common.log;

import android.util.Log;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;

/**
 * Logger provides optional logging for development purpouses.
 */
public final class Logger {

  private static final String TAG = "kontakt.io SDK";

  private static final Set<LogLevel> LOG_LEVELS = new HashSet<>();
  private static boolean DEBUG_LOGGING_ENABLED = false;
  private static boolean CRASHLYTICS_LOGGING_ENABLED = false;
  private static Method CRASHLYTICS_LOG_METHOD = null;
  private static Method CRASHLYTICS_LOG_EXCEPTION_METHOD = null;

  private Logger() {
  }

  /**
   * Enables Crashlytics logging. If your application uses Crashlytics to track thrown exceptions
   * you can easily log them via kontakt.io SDK.
   *
   * @param state the state
   */
  public static void setCrashlyticsLoggingEnabled(final boolean state) {
    if (state) {
      try {
        final Class crashlytics = Class.forName("com.crashlytics.android.Crashlytics");
        CRASHLYTICS_LOG_METHOD = crashlytics.getMethod("log", String.class);
        CRASHLYTICS_LOG_EXCEPTION_METHOD = crashlytics.getMethod("logException", Throwable.class);
        CRASHLYTICS_LOGGING_ENABLED = true;
      } catch (Exception e) {
        e.printStackTrace();
        CRASHLYTICS_LOGGING_ENABLED = false;
      }
    }
  }

  /**
   * Executes Crashlytics.log() method with applied {@link String} parameter.
   *
   * @param log the log
   */
  public static void crashlyticsLog(final String log) {
    if (CRASHLYTICS_LOGGING_ENABLED) {
      try {
        CRASHLYTICS_LOG_METHOD.invoke(null, toString(TAG, ": " + log));
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }

  /**
   * Executes Crashlytics.logException() method with applied {@link Throwable} parameter.
   *
   * @param throwable the throwable
   */
  public static void crashlyticsExceptionLog(final Throwable throwable) {
    if (CRASHLYTICS_LOGGING_ENABLED) {
      try {
        CRASHLYTICS_LOG_EXCEPTION_METHOD.invoke(null, toString(TAG, ": " + throwable));
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }

  /**
   * Enables Debug Logging. You can easily enable logging just
   * for development purpouses by checking whether BuildConfig.DEBUG
   * boolean is true.
   *
   * @param state the state
   */
  public static synchronized void setDebugLoggingEnabled(final boolean state) {
    DEBUG_LOGGING_ENABLED = state;
  }

  /**
   * Send a verbose log message.
   *
   * @param msg the message
   */
  public static void v(String msg) {
    if (DEBUG_LOGGING_ENABLED) {
      Log.v(TAG, msg);
    }
  }

  public static synchronized void setLogLevelEnabled(final LogLevel logLevel, final boolean state) {
    if (state) {
      LOG_LEVELS.add(logLevel);
    } else {
      LOG_LEVELS.remove(logLevel);
    }
  }

  public static synchronized void reset() {
    LOG_LEVELS.clear();
    DEBUG_LOGGING_ENABLED = false;
  }

  /**
   * Send a verbose log message and log the exception.
   *
   * @param msg the message
   * @param tr the throwable
   */
  public static void v(String msg, Throwable tr) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.VERBOSE)) {
      Log.v(TAG, msg, tr);
    }
  }

  /**
   * Send a Debug log message.
   *
   * @param msg the message
   */
  public static void d(String msg) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.DEBUG)) {
      Log.d(TAG, msg);
    }
  }

  /**
   * Send a Debug log message and log the exception.
   *
   * @param msg the message
   * @param tr the throwable
   */
  public static void d(String msg, Throwable tr) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.DEBUG)) {
      Log.d(TAG, msg, tr);
    }
  }

  /**
   * Send an Info log message
   *
   * @param msg the message
   */
  public static void i(String msg) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.INFO)) {
      Log.i(TAG, msg);
    }
  }

  /**
   * Send an Info log message and log the exception.
   *
   * @param msg the message
   * @param tr the throwable
   */
  public static void i(String msg, Throwable tr) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.INFO)) {
      Log.i(TAG, msg, tr);
    }
  }

  /**
   * Send a warning log message.
   *
   * @param msg the message
   */
  public static void w(String msg) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.WARNING)) {
      Log.w(TAG, msg);
    }
  }

  /**
   * Send a warning log message and log the exception.
   *
   * @param msg the message
   * @param tr the throwable
   */
  public static void w(String msg, Throwable tr) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.WARNING)) {
      Log.w(TAG, msg, tr);
    }
  }

  /**
   * Send a warning log message and log the exception.
   *
   * @param tr the throwable
   */
  public static void w(Throwable tr) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.WARNING)) {
      Log.w(TAG, tr);
    }
  }

  /**
   * Send an Error log message.
   *
   * @param msg the message
   */
  public static void e(String msg) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.ERROR)) {
      Log.e(TAG, msg);
    }
  }

  /**
   * Send an Error log message and log the exception.
   *
   * @param msg the message
   * @param tr the throwable
   */
  public static void e(String msg, Throwable tr) {
    if (DEBUG_LOGGING_ENABLED && LOG_LEVELS.contains(LogLevel.ERROR)) {
      Log.e(TAG, msg, tr);
    }
  }

  private static String toString(final String... strings) {
    if (strings == null || strings.length == 0) {
      return "";
    }

    final StringBuilder builder = new StringBuilder();
    for (final String s : strings) {
      builder.append(s);
    }

    return builder.toString();
  }
}
