package com.kontakt.sdk.android.cloud.api.executor.venues;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.VenuesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Access;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link VenuesApi}. Use this class if you want to share
 * venues to managers through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.venues().share(IDs)
 *      .toManagers(managerEmails)
 *      .withAccess(Access.VIEWER)
 *      .expirationDate(date)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that managers and access must be specified so invocations of {@code toManagers} and
 * {@code withAccess} methods are mandatory. Moreover, the {@code Access.OWNER} and
 * {@code Access.SUPERVISOR} values are prohibited as an input parameters of {@code withAccess} method.
 * All conditions must be fulfilled, otherwise an exception will be thrown at runtime.
 */
public class ShareVenueRequestExecutor extends RequestExecutor<String> {

  private final VenuesService venuesService;

  private final UUID[] ids;
  private String[] managerEmails;
  private Access access;
  private long expirationDate = -1;
  private boolean withMetaData;

  /**
   * Constructs request executor initialized with corresponding service class and venue unique IDs.
   *
   * @param venuesService the venues API service.
   * @param ids venue unique identifiers.
   */
  public ShareVenueRequestExecutor(final VenuesService venuesService, final UUID... ids) {
    this.venuesService = venuesService;
    this.ids = ids;
  }

  /**
   * Constructs request executor initialized with corresponding service class and venue unique IDs.
   *
   * @param venuesService the venues API service.
   * @param ids venue unique identifiers.
   */
  public ShareVenueRequestExecutor(final VenuesService venuesService, final List<UUID> ids) {
    this.venuesService = venuesService;
    final int size = ids.size();
    this.ids = ids.toArray(new UUID[size]);
  }

  /**
   * Specifies managers. The method invocation is obligatory while using this request executor.
   *
   * @param managerEmails managers email addresses.
   * @return this request executor.
   */
  public ShareVenueRequestExecutor toManagers(final String... managerEmails) {
    SDKPreconditions.checkNotNull(managerEmails, "manager emails cannot be null");
    this.managerEmails = managerEmails;
    return this;
  }

  /**
   * Specifies managers. The method invocation is obligatory while using this request executor.
   *
   * @param managerEmails managers email addresses.
   * @return this request executor.
   */
  public ShareVenueRequestExecutor toManagers(final List<String> managerEmails) {
    SDKPreconditions.checkNotNull(managerEmails, "manager emails cannot be null");
    final int size = managerEmails.size();
    this.managerEmails = managerEmails.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies an access. Cannot be neither {@code OWNER} nor {@code SUPERVISOR}.
   *
   * @param access the access.
   * @return this request executor.
   */
  public ShareVenueRequestExecutor withAccess(final Access access) {
    SDKPreconditions.checkNotNull(access, "access cannot be null");
    SDKPreconditions.checkState(access == Access.VIEWER || access == Access.EDITOR,
        "cannot share venue with " + access.name() + " access - should be VIEWER or EDITOR");
    this.access = access;
    return this;
  }

  /**
   * Specifies an expiration date.
   *
   * @param expirationDate the expiration date in millis.
   * @return this request executor.
   */
  public ShareVenueRequestExecutor expirationDate(final long expirationDate) {
    SDKPreconditions.checkState(expirationDate > 0, "expiration date cannot be negative");
    this.expirationDate = expirationDate;
    return this;
  }

  /**
   * Indicates if we want to share devices with metadata or not.
   *
   * @param withMetaData true or false
   * @return this request executor.
   */
  public ShareVenueRequestExecutor withMetaData(final boolean withMetaData) {
    this.withMetaData = withMetaData;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return venuesService.shareVenue(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(managerEmails != null, "cannot share venue - specify managers");
    SDKPreconditions.checkState(access != null, "cannot share venue - specify access");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Venues.VENUE_ID_PARAMETER, StringUtils.join(ids, ","));
    params.put("managerMail", StringUtils.join(managerEmails, ","));
    params.put(CloudConstants.Common.ACCESS_PARAMETER, access.name());
    params.put(CloudConstants.Venues.WITH_META_DATA_PARAMETER, String.valueOf(withMetaData));
    if (expirationDate >= 0) {
      params.put(CloudConstants.Venues.EXPIRATION_DATE_PARAMETER, String.valueOf(expirationDate));
    }
    return params;
  }
}
