package com.kontakt.sdk.android.cloud.api.executor.triggers;

import android.text.TextUtils;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.cloud.api.TriggersApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.TriggersService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Trigger;
import com.kontakt.sdk.android.common.model.TriggerContext;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.cloud.CloudConstants.Triggers.*;

/**
 * Request executor provided by {@link TriggersApi}. Use this class if you want to create
 * trigger through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *
 *   TriggerContext context = new TriggerContext.Builder()
 *      .trackingId("abcd")
 *      .build();
 *
 *   Trigger trigger = new Trigger.Builder()
 *      .name("Entered office")
 *      .context(context)
 *      .build();
 *   kontaktCloud.triggers().create(trigger).execute();
 *   </code>
 * </pre>
 */
public class CreateTriggerRequestExecutor extends RequestExecutor<Trigger> {

  private final TriggersService triggersService;
  private final Trigger trigger;

  /**
   * Constructs request executor initialized with corresponding service class and trigger object.
   *
   * @param triggersService the triggers API service.
   * @param trigger the trigger entity.
   */
  public CreateTriggerRequestExecutor(final TriggersService triggersService, final Trigger trigger) {
    this.triggersService = triggersService;
    this.trigger = trigger;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Trigger execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Trigger> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Trigger> prepareCall() {
    return triggersService.create(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(trigger.getName() != null, "Cannot create trigger - specify name");
    SDKPreconditions.checkState(trigger.getType() != null, "Cannot create trigger - specify type");
    SDKPreconditions.checkState(trigger.getContext() != null, "Cannot create trigger - specify context");
    SDKPreconditions.checkState(trigger.getExecutor() != null, "Cannot create trigger - specify executor");
    SDKPreconditions.checkState(validTrackingData(), "Cannot create trigger - specify at least one of tracking identifiers (ID, region, namespace).");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(NAME_PARAMETER, trigger.getName());
    params.put(TYPE_PARAMETER, trigger.getType().name());
    params.put(EXECUTOR_PARAMETER, trigger.getExecutor().name());
    if (trigger.getContext().getTrackingId() != null) {
      params.put(CONTEXT_TRACKING_ID_PARAMETER, trigger.getContext().getTrackingId());
    }
    if (trigger.getContext().getProximity() != null) {
      params.put(CONTEXT_PROXIMITY_PARAMETER, trigger.getContext().getProximity().name());
    }
    if (trigger.getContext().getSourceId() != null) {
      params.put(CONTEXT_SOURCE_ID_PARAMETER, trigger.getContext().getSourceId());
    }
    params.putAll(regionParams());
    params.putAll(namespaceParams());

    return params;
  }

  private boolean validTrackingData() {
    TriggerContext context = trigger.getContext();

    UUID proximityUUID = context.getProximityUUID();
    int major = context.getMajor();
    int minor = context.getMinor();

    boolean invalidMajorMinor = major == BeaconRegion.ANY_MAJOR && minor != BeaconRegion.ANY_MINOR;
    boolean validRegionData = proximityUUID != null && !invalidMajorMinor;

    String namespace = context.getNamespace();
    boolean validNamespaceData = !TextUtils.isEmpty(namespace);

    String trackingId = context.getTrackingId();
    boolean validTrackingId = !TextUtils.isEmpty(trackingId);

    return validRegionData || validNamespaceData || validTrackingId;
  }

  private Map<String, String> regionParams() {
    Map<String, String> regionParams = new HashMap<>();
    TriggerContext context = trigger.getContext();

    UUID proximityUUID = context.getProximityUUID();
    int major = context.getMajor();
    int minor = context.getMinor();

    boolean invalidMajorMinor = major == BeaconRegion.ANY_MAJOR && minor != BeaconRegion.ANY_MINOR;
    boolean validRegionData = proximityUUID != null && !invalidMajorMinor;
    if (validRegionData) {
      regionParams.put(CONTEXT_PROXIMITY_UUID_PARAMETER, proximityUUID.toString());
      if (major != BeaconRegion.ANY_MAJOR) {
        regionParams.put(CONTEXT_MAJOR_PARAMETER, String.valueOf(major));
      }
      if (minor != BeaconRegion.ANY_MINOR) {
        regionParams.put(CONTEXT_MINOR_PARAMETER, String.valueOf(minor));
      }
    }
    return regionParams;
  }

  private Map<String, String> namespaceParams() {
    Map<String, String> namespaceParams = new HashMap<>();
    TriggerContext context = trigger.getContext();

    String namespace = context.getNamespace();
    String instanceId = context.getInstanceId();

    boolean validNamespaceData = !TextUtils.isEmpty(namespace);
    if (validNamespaceData) {
      namespaceParams.put(CONTEXT_NAMESPACE_PARAMETER, namespace);
      if (!TextUtils.isEmpty(instanceId)) {
        namespaceParams.put(CONTEXT_INSTANCE_ID_PARAMETER, instanceId);
      }
    }
    return namespaceParams;
  }
}
