package com.kontakt.sdk.android.cloud.api.executor.managers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ManagersApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ManagersService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Manager;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link ManagersApi}. Use this class if you want to create
 * managers through fluent API in chained fashion, for instance:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Manager manager = new Manager.Builder()
 *      .firstName("Superb")
 *      .lastName("Kontakter")
 *      .email("superb.kontakter@kontakt.io")
 *      .role(Manager.Role.SUPERUSER)
 *      .build();
 *   kontaktCloud.managers().create(manager).execute();
 *   </code>
 * </pre>
 * Keep in mind that new manager instance that we want to create must have initialized the following
 * fields: first name, last name, email address and role. Otherwise an exception will be thrown.
 */
public class CreateManagerRequestExecutor extends RequestExecutor<Manager> {

  private final ManagersService managersService;
  private final Manager manager;

  /**
   * Constructs request executor initialized with corresponding service class and manager object.
   *
   * @param managersService the managers API service.
   * @param manager the manager entity.
   */
  public CreateManagerRequestExecutor(final ManagersService managersService, final Manager manager) {
    this.managersService = managersService;
    this.manager = manager;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Manager> prepareCall() {
    return managersService.createManager(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Manager execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Manager> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(manager.getFirstName() != null, "cannot create manager - specify first name");
    SDKPreconditions.checkState(manager.getLastName() != null, "cannot create manager - specify last name");
    SDKPreconditions.checkState(manager.getEmail() != null, "cannot create manager - specify email");
    SDKPreconditions.checkState(manager.getRole() != null, "cannot create manager - specify role");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Managers.FIRST_NAME_PARAMETER, manager.getFirstName());
    params.put(CloudConstants.Managers.LAST_NAME_PARAMETER, manager.getLastName());
    params.put(CloudConstants.Managers.EMAIL_PARAMETER, manager.getEmail());
    params.put(CloudConstants.Managers.ROLE_PARAMETER, manager.getRole().name());

    return params;
  }
}
