package com.kontakt.sdk.android.cloud.api.executor.firmwares;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.FirmwaresApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.FirmwaresService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.paginated.Firmwares;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.FirmwareType;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import retrofit2.Call;

/**
 * Request executor provided by {@link FirmwaresApi}. Use this class if you want to get
 * firmwares through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Firmwares firmwares = kontaktCloud.firmwares().fetch()
 *      .forDevices("4DgT", "6Hyj")
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that devices must be specified so invocation of {@code forDevices} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class FirmwaresRequestExecutor extends PaginatedRequestExecutor<Firmwares> {

  private final FirmwaresService firmwaresService;
  private String[] deviceUniqueIds;
  private FirmwareType type;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param firmwaresService the firmwares API service.
   */
  public FirmwaresRequestExecutor(final FirmwaresService firmwaresService) {
    this.firmwaresService = firmwaresService;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public FirmwaresRequestExecutor forDevices(final String... uniqueIds) {
    this.deviceUniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public FirmwaresRequestExecutor forDevices(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.deviceUniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public FirmwaresRequestExecutor forDevices(final Set<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.deviceUniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies firmware type.
   *
   * @param type the firmware type.
   * @return this request executor.
   */
  public FirmwaresRequestExecutor withType(final FirmwareType type) {
    SDKPreconditions.checkNotNull(type, "type cannot be null");
    this.type = type;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FirmwaresRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FirmwaresRequestExecutor maxResult(int maxResult) {
    super.startIndex(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FirmwaresRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FirmwaresRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FirmwaresRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Firmwares execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<Firmwares> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Firmwares> prepareCall() {
    final Call<Firmwares> call;
    if (eTag != null) {
      call = firmwaresService.getFirmwares(params(), eTag);
    } else {
      call = firmwaresService.getFirmwares(params());
    }
    return call;
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(deviceUniqueIds != null, "cannot get firmwares - specify device IDs");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (deviceUniqueIds != null) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(deviceUniqueIds, ","));
    }
    if (type != null) {
      params.put(CloudConstants.Firmwares.TYPE_PARAMETER, type.name());
    }
    return params;
  }
}
