package com.kontakt.sdk.android.cloud.api.executor.firmwares;

import com.kontakt.sdk.android.cloud.api.FirmwaresApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.FirmwaresService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link FirmwaresApi}. Use this class if you want to get
 * single firmware through fluent API in chained fashion, for instance:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   String name = "3.1";
 *   Firmware firmware = kontaktCloud.firmwares().fetch(name)
 *      .forDeviceType(DeviceType.BEACON)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that device type must be specified so invocation of {@code forDeviceType} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class FirmwareRequestExecutor extends RequestExecutor<Firmware> {

  private final FirmwaresService firmwaresService;

  private final String name;
  private DeviceType deviceType;

  /**
   * Constructs request executor initialized with corresponding service class and firmware name.
   *
   * @param firmwaresService the firmwares API service.
   * @param name the firmware name.
   */
  public FirmwareRequestExecutor(final FirmwaresService firmwaresService, final String name) {
    this.firmwaresService = firmwaresService;
    this.name = name;
  }

  /**
   * Specifies device type. The method invocation is obligatory while using this request executor.
   *
   * @param deviceType the device type.
   * @return this request executor.
   */
  public FirmwareRequestExecutor forDeviceType(final DeviceType deviceType) {
    SDKPreconditions.checkNotNull(deviceType, "device type cannot be null");
    this.deviceType = deviceType;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Firmware execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<Firmware> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(deviceType != null, "cannot get firmware - specify device type");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Firmware> prepareCall() {
    return firmwaresService.getFirmware(name, deviceType.name());
  }

  @Override
  protected Map<String, String> params() {
    return new HashMap<>();
  }
}
